// SPDX-License-Identifier: MIT


#define _GNU_SOURCE  // tdestroy()

#include <febug-abi.h>
#include <libfebug.h>

#include <sys/socket.h>
#include <sys/types.h>
#include <sys/un.h>

#include <inttypes.h>
#include <unistd.h>

#include <errno.h>
#include <search.h>
#include <stdlib.h>
#include <string.h>


#if __APPLE__
// GUESS WHAT?? DARWIN DEFINES SOCK_SEQPACKET BUT DOESN'T ACTUALLY FUCKING SUPPORT IT (i.e. socket(2) returns EPROTONOSUPPORT). WHY? BECAUSE FUCK YOU.
#undef SOCK_SEQPACKET
#define SOCK_SEQPACKET SOCK_STREAM
#endif


struct formatter_t {
	uint64_t type;
	void (*formatter)(int, size_t);
};

static int formatter_t_compare(const void * lhs, const void * rhs) {
	return ((const struct formatter_t *)lhs)->type - ((const struct formatter_t *)rhs)->type;
}


int febug_global_controlled_socket = -1;

static void * formatters = 0;


void febug_start_path(const char * path) {
	if(getenv("FEBUG_DONT"))
		return;

	if((febug_global_controlled_socket = socket(AF_UNIX, SOCK_SEQPACKET, 0)) == -1) {
		fprintf(stderr, "febug_start_path: socket: %s\n", strerror(errno));
		return;
	}

	const char * pe = getenv("FEBUG_SOCKET");
	if(pe)
		path = pe;

	struct sockaddr_un addr = {0};
	addr.sun_family         = AF_UNIX;
	(void)strncpy(addr.sun_path, path, sizeof(addr.sun_path) - 1);
	if(connect(febug_global_controlled_socket, (const struct sockaddr *)&addr, sizeof(addr)) == -1) {
		fprintf(stderr, "febug_start_path: connect: %s\n", strerror(errno));
		close(febug_global_controlled_socket);
		febug_global_controlled_socket = -1;
		return;
	}

#if __linux__ || __OpenBSD__ || __APPLE__
	// Handled automatically with SO_PASSCRED, also the manual variant didn't work for some reason
	// Only way is getsockopt(SO_PEERCRED)
	// Only way is getsockopt(LOCAL_PEERCRED)+getsockopt(LOCAL_PEEREPID)
#elif __NetBSD__
	// Correct way is automatically via LOCAL_CREDS
	// However, the message /must/ be sent after the peer sets it; use a sync message from the server for this,
	// otherwise we sent the first febug_message too quickly sometimes
	struct attn_febug_message sync_msg;
	if(recv(febug_global_controlled_socket, &sync_msg, sizeof(sync_msg), 0) != sizeof(sync_msg)) {
		fprintf(stderr, "febug_start_path: recv: %s\n", strerror(errno));
		close(febug_global_controlled_socket);
		febug_global_controlled_socket = -1;
		return;
	}
#else
	// From FreeBSD 12.1-RELEASE-p7 /usr/include/socket.h:
	/*
	 * Credentials structure, used to verify the identity of a peer
	 * process that has sent us a message. This is allocated by the
	 * peer process but filled in by the kernel. This prevents the
	 * peer from lying about its identity. (Note that cmcred_groups[0]
	 * is the effective GID.)
	 */
	char cmsgbuf[CMSG_SPACE(sizeof(struct cmsgcred))];

	struct msghdr msg;
	memset(&msg, 0, sizeof(msg));
	msg.msg_control    = cmsgbuf;
	msg.msg_controllen = sizeof(cmsgbuf);

	struct cmsghdr * cmsg = CMSG_FIRSTHDR(&msg);
	cmsg->cmsg_level      = SOL_SOCKET;
	cmsg->cmsg_type       = SCM_CREDS;
	cmsg->cmsg_len        = CMSG_LEN(sizeof(struct cmsgcred));
	msg.msg_controllen    = cmsg->cmsg_len;  // total size of all control blocks

	if((sendmsg(febug_global_controlled_socket, &msg, 0)) == -1) {
		fprintf(stderr, "febug_start_path: sendmsg: %m\n");
		close(febug_global_controlled_socket);
		febug_global_controlled_socket = -1;
		return;
	}
#endif
}

void febug_end() {
	if(febug_global_controlled_socket != -1) {
		close(febug_global_controlled_socket);
		febug_global_controlled_socket = -1;
	}

#if __linux__  // Extension, but musl supports it
	if(formatters) {
		tdestroy(formatters, free);
		formatters = 0;
	}
#else  // https://stackoverflow.com/a/39569954/2851815
	while(formatters) {
		struct formatter_t * root_data = *(struct formatter_t **)formatters;
		tdelete(root_data, &formatters, formatter_t_compare);
		free(root_data);
	}
#endif
}


void febug_register_type(uint64_t type, void (*formatter)(int, size_t)) {
	if(febug_global_controlled_socket == -1)
		return;

	struct formatter_t * new = calloc(1, sizeof(*new));
	if(!new)
		return;
	new->type      = type;
	new->formatter = formatter;

	if(!tsearch(new, &formatters, formatter_t_compare))
		free(new);
}


void febug_debug_handler(int _) {
	(void)_;
	if(febug_global_controlled_socket == -1)
		return;

	struct attn_febug_message afmsg = {0};
	struct iovec buf_i              = {&afmsg, sizeof(afmsg)};

	int retpipe = -1;
	char cmsgbuf[CMSG_SPACE(sizeof(retpipe))];

	struct msghdr msg  = {0};
	msg.msg_iov        = &buf_i;
	msg.msg_iovlen     = 1;
	msg.msg_control    = cmsgbuf;
	msg.msg_controllen = sizeof(cmsgbuf);

	if(recvmsg(febug_global_controlled_socket, &msg, 0) == -1) {
		fprintf(stderr, "febug_debug_handler: recvmsg: %s\n", strerror(errno));
		return;
	}

	const struct cmsghdr * cmsg = CMSG_FIRSTHDR(&msg);
	if(cmsg && cmsg->cmsg_type == SCM_RIGHTS) {
		memcpy(&retpipe, CMSG_DATA(cmsg), sizeof(retpipe));

		struct formatter_t looking_for  = {.type = afmsg.variable_type};
		const struct formatter_t ** itr = tfind(&looking_for, &formatters, formatter_t_compare);
		if(itr)
			(*itr)->formatter(retpipe, afmsg.variable_id);
		else
			dprintf(retpipe, "Unknown variable type %" PRIu64 " with ID %" PRIu64 "\n", afmsg.variable_type, afmsg.variable_id);

		close(retpipe);
	} else
		fprintf(stderr, "febug_debug_handler: cmsg: no fd\n");
}


void febug_wrap_signalv(uint64_t type, const void * data, uint8_t signal, const char * name, va_list ap) {
	if(febug_global_controlled_socket != -1) {
		struct febug_message msg = {.variable_id = (uint64_t)(size_t)data, .variable_type = type, .signal = signal};
		vsnprintf(msg.name, sizeof(msg.name), name, ap);
		send(febug_global_controlled_socket, &msg, sizeof(msg), 0);
	}
}

void febug_unwrap(const void * data) {
	if(febug_global_controlled_socket != -1) {
		struct stop_febug_message msg = {.variable_id = (uint64_t)(size_t)data};
		send(febug_global_controlled_socket, &msg, sizeof(msg), 0);
	}
}
