// SPDX-License-Identifier: MIT


#pragma once


#include <signal.h>
#include <sys/socket.h>
#include <sys/types.h>

#include <cstdio>
#include <map>
#include <stdarg.h>

#include <type_traits>
#include <typeinfo>

#include "febug-abi.h"


// Documented in libfebug(3)


#if FEBUG_DONT
#define FEBUG_MAYBE(...) static
#else
#define FEBUG_MAYBE(...) __VA_ARGS__
#endif

#ifndef FEBUG_SIGNUM
#define FEBUG_SIGNUM SIGUSR2
#endif

#ifndef FEBUG_SOCKET
#if __linux__
#define FEBUG_SOCKET "/run/febug.sock"
#else
#define FEBUG_SOCKET "/var/run/febug.sock"
#endif
#endif


namespace febug {
#if FEBUG_DONT
	namespace {
#endif
		struct controlled_socket {
			int fd = -1;

			inline operator int() const noexcept { return this->fd; }

			controlled_socket(const char * path = FEBUG_SOCKET) noexcept;
			~controlled_socket() noexcept;
		};
		FEBUG_MAYBE(extern) controlled_socket global_controlled_socket;


		template <class T>
		struct wrapper {
			const T * data;

			__attribute__((format(printf, 3, 4))) wrapper(const T & data, const char * name, ...) noexcept : data(&data) {
				va_list ap;
				va_start(ap, name);
				wrapper_impl(FEBUG_SIGNUM, name, ap);
				va_end(ap);
			}

			__attribute__((format(printf, 4, 5))) wrapper(const T & data, uint8_t signal, const char * name, ...) noexcept : data(&data) {
				va_list ap;
				va_start(ap, name);
				wrapper_impl(signal, name, ap);
				va_end(ap);
			}

			__attribute__((format(printf, 4, 0))) wrapper(const T & data, uint8_t signal, const char * name, va_list ap) noexcept : data(&data) {
				wrapper_impl(signal, name, ap);
			}

			~wrapper() noexcept {
#if !FEBUG_DONT
				if(global_controlled_socket != -1) {
					stop_febug_message msg{.variable_id = (uint64_t)this->data};
					send(global_controlled_socket, &msg, sizeof(msg), 0);
				}
#endif
			}

		private:
			void wrapper_impl(uint8_t signal, const char * name, va_list ap) noexcept {
#if !FEBUG_DONT
				if(global_controlled_socket != -1) {
					febug_message msg{.variable_id = (uint64_t)this->data, .variable_type = typeid(std::decay_t<T>).hash_code(), .signal = signal};
					std::vsnprintf(msg.name, sizeof(msg.name), name, ap);
					send(global_controlled_socket, &msg, sizeof(msg), 0);
				}
#endif
			}
		};


		FEBUG_MAYBE(extern) std::map<size_t, void (*)(int, size_t)> formatters;
		void debug_handler(int) noexcept;

#if FEBUG_DONT
		controlled_socket::controlled_socket(const char *) noexcept {}
		controlled_socket::~controlled_socket() noexcept {}

		void debug_handler(int) noexcept {}
	}
#endif
}
