/* SPDX-License-Identifier: 0BSD */


#include "../fd.hpp"
#include "../fido2.hpp"
#include "../main.hpp"
#include "../zfs.hpp"
#include "clear-key.hpp"


#define THIS_BACKEND "FIDO2"


int main(int argc, char ** argv) {
	const char * backup{};
	return do_main(
	    argc, argv, "b:", gettext_noop("[-b backup-file]"), [&](auto) { backup = optarg; },
	    [&](auto dataset) {
		    REQUIRE_KEY_LOADED(dataset);

		    fido_init(0);
		    TRY_MAIN(verify_backend(dataset, THIS_BACKEND, [&](auto previous_handle) {
			    clear_key_bundle dt{};
			    if(!zfs_fido2_clear_key_handle(dt, dataset, previous_handle))
				    zfs_fido2_clear_key_free(dt);
		    }));

		    fido2_device dev;
		    TRY_MAIN(fido2_find_device(dev));


		    fido2_cred_bundle cred;
		    TRY_MAIN(fido2_genkey(cred, zfs_get_name(dataset), false, dev));
		    bool ok = false;  // Try to free the credential if we're unsuccessful in actually using it later on
		    quickscope_wrapper cred_clearer{[&] {
			    if(!ok) {
				    zfs_fido2_clear_key_free_one(dev, zfs_get_name(dataset), cred);
				    clear_key_props(dataset);
			    }
		    }};

		    TRY("getentropy", getentropy(cred.salt, sizeof(cred.salt)));
		    uint8_t wrap_key[WRAPPING_KEY_LEN];
		    TRY_MAIN(fido2_loadkey(wrap_key, zfs_get_name(dataset), false, dev, cred));

		    if(backup)
			    TRY_MAIN(write_exact(backup, wrap_key, sizeof(wrap_key), 0400));


		    {
			    char * prop{};
			    TRY_MAIN(fido2_unparse_prop(prop, cred, nullptr, 0));
			    quickscope_wrapper prop_deleter{[&] { free(prop); }};
			    TRY_MAIN(set_key_props(dataset, THIS_BACKEND, prop));
		    }

		    TRY_MAIN(change_key(dataset, wrap_key));

		    ok = true;
		    return 0;
	    });
}
