/* SPDX-License-Identifier: 0BSD */


#pragma once


#include "common.hpp"
#include <fido.h>
#include <fido/credman.h>
#include <fido/es256.h>


#define TRY_FIDO2(what, ...) TRY_GENERIC(what, , != FIDO_OK, _try_ret, __LINE__, fido_strerr, __VA_ARGS__)

// https://sources.debian.org/src/pam-u2f/1.1.0-1.1/pamu2fcfg/pamu2fcfg.c/?hl=168#L59
// https://github.com/hoytech/defido2/blob/2149746d2bcc9c177f802e1d3ddb706931c1a44d/cmd_init.cpp#L57
// idk
#define FIDO2_UID_LEN 32

// https://sources.debian.org/src/pam-u2f/1.1.0-1.1/pamu2fcfg/pamu2fcfg.c/?hl=168#L60
// idk
#define FIDO2_CLIENTDATAHASH_LEN 32

// https://docs.yubico.com/yesdk/users-manual/application-fido2/hmac-secret.html
#define FIDO2_SALT_LEN 32

#define FIDO2_HMAC_SECRET_RESULT_LEN 32


#define FIDO_MAXDEVS 64  // convenience; could bump


struct fido2_device {
	fido_dev_t * dev;
	char * name;
	bool has_del;

	constexpr operator fido_dev_t *() noexcept { return this->dev; }
};
extern int fido2_find_device(fido2_device & ret, int (*shop)(fido2_device & ret, bool & ret_ret, void * dt) = nullptr, void * dt = nullptr);
extern int fido2_enumerate_devices(int (*visit)(fido2_device & device, bool & ret_ret, bool & deldev, void * dt), void * dt);


struct fido2_cred_bundle {
	uint8_t salt[FIDO2_SALT_LEN];
	const uint8_t *cred_id, *pubkey;
	size_t cred_id_len, pubkey_len;

	char * cred_id_str() const noexcept;
};
#define ENCRYPTED_AES_IV_LEN 12  // GCM
#define ENCRYPTED_LEN 32         // WRAPPING_KEY_LEN after GCM
struct fido2_backup_cred_bundle {
	fido2_cred_bundle cred;
	uint8_t iv[ENCRYPTED_AES_IV_LEN];
	uint8_t encrypted[ENCRYPTED_LEN];
};
extern int fido2_genkey(fido2_cred_bundle & ret, const char * dataset, bool backup, fido2_device & dev);
extern int fido2_loadkey(uint8_t (&ret)[FIDO2_HMAC_SECRET_RESULT_LEN], const char * dataset, bool backup, fido2_device & dev, const fido2_cred_bundle & bundle,
                         bool * fatal = nullptr);

extern int fido2_unparse_prop(char *& prop, const fido2_cred_bundle & bundle, const fido2_backup_cred_bundle * backups, size_t backups_len);
extern int fido2_parse_prop(fido2_cred_bundle & bundle, fido2_backup_cred_bundle *& backups, size_t & backups_len, const char * dataset_name, char * handle_s);


extern int fido2ify_passphrase(char *& pin, const uint8_t * buf, size_t len);
extern int fido2_prompt_pin(char *& pin, const fido2_device & dev);
