/* SPDX-License-Identifier: 0BSD */


#include <algorithm>
#include <stdio.h>

#include "../fido2.hpp"
#include "../main.hpp"
#include "../openssl.hpp"
#include "../zfs.hpp"
#include "load-key.hpp"


#define THIS_BACKEND "FIDO2"


int main(int argc, char ** argv) {
	return do_main(
	    argc, argv, "", "", [](auto) {},
	    [&](auto dataset) {
		    char * handle_s{};
		    TRY_MAIN(parse_key_props(dataset, THIS_BACKEND, handle_s));

		    struct add_backup_bundle : load_key_bundle {
			    fido2_device new_backup_dev;
			    struct {
				    bool primary;
				    bool * backups;
			    } got;
		    } dt{};
		    dt.dataset_name = zfs_get_name(dataset);
		    TRY_MAIN(fido2_parse_prop(dt.bundle, dt.backups, dt.backups_len, zfs_get_name(dataset), handle_s));
		    dt.got.backups = TRY_PTR("allocate backup mask", reinterpret_cast<bool *>(calloc(dt.backups_len, sizeof(*dt.got.backups))));


		    fido_init(0);

		    TRY_MAIN(fido2_enumerate_devices(
		        [](fido2_device & dev, bool & ret_ret, bool & deldev, void * dt_raw) -> int {
			        auto & dt = *reinterpret_cast<add_backup_bundle *>(dt_raw);
			        deldev    = true;

			        if(!dt.got.primary) {
				        UNLOCK_PRIMARY(ret_ret        = false;  //
				                       dt.got.primary = true;   //
				                       goto end;)
			        }
			        for(size_t i = 0; i < dt.backups_len; ++i) {
				        if(dt.got.backups[i])
					        continue;

				        UNLOCK_BACKUP()
				        dt.got.backups[i] = true;
				        goto end;
			        }

			        if(!dt.new_backup_dev) {
				        dt.new_backup_dev = dev;
				        deldev            = false;
			        }

		        end:
			        return !(dt.new_backup_dev && dt.got.primary && std::all_of(dt.got.backups, dt.got.backups + dt.backups_len, [](auto m) { return m; }));
		        },
		        &dt));

		    if(!dt.got.primary && !std::any_of(dt.got.backups, dt.got.backups + dt.backups_len, [](auto m) { return m; }))
			    return fprintf(stderr, gettext("No device with key for %s found!\n"), dt.dataset_name), __LINE__;
		    if(!dt.new_backup_dev)
			    return fputs(gettext("No eligible FIDO2 devices!\n"), stderr), __LINE__;


		    dt.backups = TRY_PTR("allocate backup", reinterpret_cast<fido2_backup_cred_bundle *>(reallocarray(dt.backups, ++dt.backups_len, sizeof(*dt.backups))));
		    auto & backup_cred = *new(&dt.backups[dt.backups_len - 1]) fido2_backup_cred_bundle;

		    TRY("getentropy", getentropy(backup_cred.cred.salt, sizeof(backup_cred.cred.salt)));
		    TRY_MAIN(fido2_genkey(backup_cred.cred, zfs_get_name(dataset), true, dt.new_backup_dev));

		    uint8_t sym_key[AES_KEY_LEN];
		    TRY_MAIN(fido2_loadkey(sym_key, zfs_get_name(dataset), true, dt.new_backup_dev, backup_cred.cred));


		    TRY("getentropy", getentropy(backup_cred.iv, sizeof(backup_cred.iv)));
		    TRY_MAIN(encrypt_backup(backup_cred.encrypted, sym_key, backup_cred.iv, dt.wrap_key));


		    char * prop{};
		    TRY_MAIN(fido2_unparse_prop(prop, dt.bundle, dt.backups, dt.backups_len));
		    TRY_MAIN(set_key_props(dataset, THIS_BACKEND, prop));
		    return 0;
	    });
}
