/* SPDX-License-Identifier: 0BSD */


#include "bin/clear-key.hpp"
#include "zfs.hpp"
#include <algorithm>


int zfs_fido2_clear_key_handle(clear_key_bundle & dt, zfs_handle_t * dataset, char * handle_s) {
	dt.dataset_name = zfs_get_name(dataset);
	TRY_MAIN(fido2_parse_prop(dt.bundle, dt.backups, dt.backups_len, zfs_get_name(dataset), handle_s));
	dt.found.backups = TRY_PTR("allocate found mask", reinterpret_cast<bool *>(calloc(dt.backups_len, sizeof(*dt.found.backups))));
	return 0;
}

static int free_cb(fido2_device & dev, bool &, bool & deldev, void * dt_raw) {
	auto & dt = *reinterpret_cast<clear_key_bundle *>(dt_raw);
	deldev    = true;

	char * pin{};
	quickscope_wrapper pin_deleter{[&] { free(pin); }};

	auto attempt = [&](const fido2_cred_bundle & bundle, size_t backup, bool & found) {
		if(found)
			return -1;
		if(uint8_t blob[FIDO2_HMAC_SECRET_RESULT_LEN]; fido2_loadkey(blob, dt.dataset_name, backup, dev, bundle) != FIDO_OK)
			return -1;
		found = true;

		if(!dev.has_del)
			return 0;
		char * cred_id_str{};
		quickscope_wrapper cred_id_str_deleter{[&] { free(cred_id_str); }};
		if(!fido_dev_has_pin(dev)) {
			if(backup)
				fprintf(stderr,
				        gettext("Backup device %zu (%s) supports credential management but has no PIN set, so the credential cannot be removed; "
				                "remove with fido2-token -D -i %s /dev/hidrawX\n"),
				        backup, dev.name, cred_id_str = bundle.cred_id_str());
			else
				fprintf(stderr,
				        gettext("%s supports credential management but has no PIN set, so the credential cannot be removed; "
				                "remove with fido2-token -D -i %s /dev/hidrawX\n"),
				        dev.name, cred_id_str = bundle.cred_id_str());
			return 0;
		}

		TRY_MAIN(fido2_prompt_pin(pin, dev));
		if([&] { return TRY_FIDO2("delete credential", fido_credman_del_dev_rk(dev, bundle.cred_id, bundle.cred_id_len, pin)), 0; }()) {
			if(backup)
				fprintf(stderr, gettext("Failed backup credential %zu may be freed with fido2-token -D -i %s /dev/hidrawX\n"), backup,
				        cred_id_str = bundle.cred_id_str());
			else
				fprintf(stderr, gettext("Failed credential may be freed with fido2-token -D -i %s /dev/hidrawX\n"), cred_id_str = bundle.cred_id_str());
		}
		return 0;
	};

	if(!attempt(dt.bundle, false, dt.found.primary))
		return -1;
	for(size_t i = 0; i < dt.backups_len; ++i)
		if(!attempt(dt.backups[i].cred, i + 1, dt.found.backups[i]))
			return -1;

	return !(dt.found.primary && std::all_of(dt.found.backups, dt.found.backups + dt.backups_len, [](auto m) { return m; }));
}
static void diag(const fido2_cred_bundle & bundle, size_t backup, bool found) {
	if(found)
		return;

	char * cred_id_str{};
	if(backup)
		fprintf(stderr, gettext("Device with backup credential %zu not found; remove with fido2-token -D -i %s /dev/hidrawX\n"), backup,
		        cred_id_str = bundle.cred_id_str());
	else
		fprintf(stderr, gettext("Device with credential not found; remove with fido2-token -D -i %s /dev/hidrawX\n"), cred_id_str = bundle.cred_id_str());
	free(cred_id_str);
}
void zfs_fido2_clear_key_free(clear_key_bundle & dt) {
	fido2_enumerate_devices(free_cb, &dt);

	diag(dt.bundle, false, dt.found.primary);
	for(size_t i = 0; i < dt.backups_len; ++i)
		diag(dt.backups[i].cred, i + 1, dt.found.backups[i]);
}
void zfs_fido2_clear_key_free_one(fido2_device & device, const char * dataset_name, const fido2_cred_bundle & bundle) {
	bool _;
	clear_key_bundle dt{};
	dt.dataset_name = dataset_name;
	dt.bundle       = bundle;
	free_cb(device, _, _, &dt);
	diag(dt.bundle, false, dt.found.primary);
}

clear_key_bundle::~clear_key_bundle() {
	free(this->found.backups);
}
