// The MIT License (MIT)

// Copyright (c) 2020 наб <nabijaczleweli@nabijaczleweli.xyz>

// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


#include "config.hpp"
#include "context.hpp"
#include "context_detail.hpp"
#include "quickscope_wrapper.hpp"
#include <algorithm>
#include <stdlib.h>
extern "C" {
#include <efivar/efivar.h>
}


#define TRY(...)                                       \
	({                                                   \
		auto ret = __VA_ARGS__;                            \
		if(auto err = std::get_if<std::string>(&ret); err) \
			return std::move(*err);                          \
		std::move(std::get<0>(ret));                       \
	})

#define TRY_OPT(...)              \
	if(auto err = __VA_ARGS__; err) \
		return err;


struct fresh_kernel {
	std::string_view version;
	std::pair<std::string_view, std::string_view> image;
	std::vector<std::pair<std::string_view, std::string_view>> initrds;
};

namespace {
	static std::variant<klapki::context::detail::bad_cow, std::string> unrelativise(const std::pair<std::string_view, std::string_view> & whom,
	                                                                                const char * version, const char * for_what) {
		if(whom.first[0] != '/') {
			fmt::print(stderr, "{} {} has relative path {}/{}", version, for_what, whom.first, whom.second);

			std::string reldir{whom.first};
			const auto dir = realpath(reldir.c_str(), nullptr);
			if(!dir) {
				fmt::print(stderr, "\n");
				return fmt::format("Getting realpath for {}: {}", reldir, strerror(errno));
			}
			klapki::quickscope_wrapper dir_deleter{[&] { std::free(dir); }};

			fmt::print(stderr, ", updating to {}/{}\n", dir, whom.second);
			return klapki::context::detail::bad_cow{std::string{dir}};
		} else
			return klapki::context::detail::bad_cow{whom.first};
	}

	/// Actually just find lowest unoccupied
	static std::variant<std::uint16_t, std::string> allocate_bootnum(const klapki::state::state & state) {
		std::uint16_t max{};
		if(auto found = adjacent_find(std::begin(state.entries), std::end(state.entries),
		                              [&](auto && prev, auto && next) {
			                              max = next.first;
			                              return next.first - prev.first != 1;
		                              });
		   found != std::end(state.entries))
			return static_cast<std::uint16_t>(found->first + 1);
		else if(max != 0xFFFF)
			return static_cast<std::uint16_t>(max + 1);
		else
			return "Out of boot entries";
	}

	static void append_bootorder_entry(klapki::state::boot_order_t & bord, std::uint16_t bootnum) {
		std::visit(klapki::overload{
		               [&](klapki::state::boot_order_flat &) { throw __func__; },  // can't (shouldn't) happen; not like we can do much with this
		               [&](klapki::state::boot_order_structured & bos) {
			               if(!bos.order.empty())
				               if(bos.order.back().second == true) {
					               bos.order.back().first.emplace_back(bootnum);
					               return;
				               }

			               bos.order.emplace_back(std::pair{std::vector{bootnum}, true});
		               },
		           },
		           bord);
	}

	static std::vector<std::pair<klapki::state::nonbase_dirname_t, klapki::state::shaa_t>>
	compact_initrd_dirs(std::string_view prev_dir, std::vector<klapki::context::detail::bad_cow> initrd_dirs) {
		std::vector<std::pair<klapki::state::nonbase_dirname_t, klapki::state::shaa_t>> ret;
		ret.reserve(initrd_dirs.size());

		for(auto && idir : initrd_dirs) {
			if(idir.get() == prev_dir)
				ret.emplace_back(klapki::state::nonbase_dirname_t{}, klapki::state::shaa_t{});
			else
				ret.emplace_back(idir.get(), klapki::state::shaa_t{});

			prev_dir = idir.get();
		}

		return ret;
	}
}


#define COPY_SHA(s) \
	{ s[0], s[1], s[2], s[3], s[4], s[5], s[6], s[7], s[8], s[9], s[10], s[11], s[12], s[13], s[14], s[15], s[16], s[17], s[18], s[19] }

std::optional<std::string> klapki::context::context::allocate_kernel_variant(const config & cfg, state::state & state, std::string version, std::string var,
                                                                             std::string kernel_dirname, state::sha_t & kernel_image_sha,
                                                                             std::vector<std::pair<state::nonbase_dirname_t, state::shaa_t>> initrd_dirnames,
                                                                             std::string image_basename,
                                                                             std::vector<std::pair<state::nonbase_dirname_t, std::string>> initrd_paths) {
	auto efi_base = fmt::format("{}{}\\", cfg.news_efi_dir(), version);

	const auto new_bootnum = TRY(allocate_bootnum(state));
	if(cfg.verbose) {
		if(var.empty())
			fmt::print("  Default variant");
		else
			fmt::print("  Variant {}", var);
		fmt::print(" assigned bootnum {:04X}\n", new_bootnum);
	}


	append_bootorder_entry(state.order, new_bootnum);
	state.statecfg.wanted_entries.emplace_back(state::stated_config_entry{
	    new_bootnum, {0xFF}, std::move(version), std::move(var), std::move(kernel_dirname), COPY_SHA(kernel_image_sha), std::move(initrd_dirnames)});

	state.entries.emplace(new_bootnum,
	                      state::boot_entry{{}, 0, {0x00}, EFI_VARIABLE_NON_VOLATILE | EFI_VARIABLE_BOOTSERVICE_ACCESS | EFI_VARIABLE_RUNTIME_ACCESS});
	this->our_kernels.emplace(new_bootnum, our_kernel{"", "", {std::move(efi_base), std::move(image_basename)}, std::move(initrd_paths)});

	return {};
}

void klapki::context::context::purge_allocations_impl(state::state & state, const state::stated_config_entry & skern) {
	if(!state.entries.erase(skern.bootnum_hint))
		fmt::print(stderr, "Deleted entry {:04X} not in boot entries?\n", skern.bootnum_hint);

	auto kern_n = this->our_kernels.extract(skern.bootnum_hint);
	if(!kern_n)
		throw __func__;
	auto && kern = kern_n.mapped();

	std::string_view dprev = this->deleted_files.emplace(std::move(kern.image_path)).first->first;
	for(auto && [didir, dibase] : kern.initrd_paths)
		if(didir)
			dprev = this->deleted_files.emplace(std::move(*didir), std::move(dibase)).first->first;
		else
			this->deleted_files.emplace(dprev, std::move(dibase));

	std::visit(klapki::overload{
	               [&](klapki::state::boot_order_flat &) { throw __func__; },
	               [&](klapki::state::boot_order_structured & bord) {
		               for(auto && [chunk, ours] : bord.order) {
			               if(!ours)
				               continue;

			               if(auto itr = std::find(std::begin(chunk), std::end(chunk), skern.bootnum_hint); itr != std::end(chunk)) {
				               chunk.erase(itr);
				               return;
			               }
		               }

		               fmt::print(stderr, "Deleted entry {:04X} not in boot order?\n", skern.bootnum_hint);
	               },
	           },
	           state.order);
}


std::optional<std::string> klapki::context::context::age(const config & cfg, state::state & state) {
	std::vector<fresh_kernel> fkerns;
	fkerns.swap(this->fresh_kernels);
	for(auto && fkern : std::move(fkerns)) {
		if(cfg.verbose)
			fmt::print("Aging fresh kernel {}\n", fkern.version);

		const auto image_dir = TRY(unrelativise(fkern.image, fkern.version.data(), "image"));

		std::vector<detail::bad_cow> initrd_dirs_raw;
		initrd_dirs_raw.reserve(fkern.initrds.size());
		for(auto && initrd : fkern.initrds)
			initrd_dirs_raw.emplace_back(TRY(unrelativise(initrd, fkern.version.data(), "initrd")));
		const auto initrd_dirs = compact_initrd_dirs(image_dir.get(), std::move(initrd_dirs_raw));

		std::vector<std::pair<state::nonbase_dirname_t, std::string>> our_initrd_paths;
		our_initrd_paths.reserve(fkern.initrds.size());
		for(auto && initrd : fkern.initrds)
			our_initrd_paths.emplace_back(std::nullopt, initrd.second);

		state::sha_t nilsha{0x00};
		TRY_OPT(this->allocate_kernel_variant(cfg, state, std::string{fkern.version}, "", std::string{image_dir.get()}, nilsha, initrd_dirs,
		                                      std::string{fkern.image.second}, our_initrd_paths));
		for(auto && var : state.statecfg.variants)
			TRY_OPT(this->allocate_kernel_variant(cfg, state, std::string{fkern.version}, var, std::string{image_dir.get()}, nilsha, initrd_dirs,
			                                      std::string{fkern.image.second}, our_initrd_paths));
	}

	return {};
}
