// The MIT License (MIT)

// Copyright (c) 2020 наб <nabijaczleweli@nabijaczleweli.xyz>

// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


#include "config.hpp"
#include "context.hpp"
#include "context_detail.hpp"
#include <fmt/format.h>
#include <string_view>
extern "C" {
#include <efivar/efiboot.h>
#include <efivar/efivar-dp.h>
#include <ucs2.h>
}


namespace {
	/// unique_ptr deleter that uses free(3)
	struct free_deleter {
		template <class T>
		void operator()(T * ptr) const noexcept {
			std::free(ptr);
		}
	};
}


klapki::context::context klapki::context::context::derive(const config & cfg, state::state & output_state) {
	context ret{};

	for(auto && went : output_state.statecfg.wanted_entries)
		if(auto bent = output_state.entries.find(went.bootnum_hint); bent != std::end(output_state.entries)) {
			our_kernel kern{};

			const auto efi_opt        = reinterpret_cast<efi_load_option *>(bent->second.load_option.get());
			const ssize_t efi_opt_len = bent->second.load_option_len;
			if(!efi_loadopt_is_valid(efi_opt, efi_opt_len)) {
				fmt::print(stderr, "Entry {:04X} not a valid EFI Load Option. Abandoning.\n", went.bootnum_hint);
				continue;
			}

			const auto dp = efi_loadopt_path(efi_opt, efi_opt_len);
			if(dp->type != EFIDP_MEDIA_TYPE && dp->subtype != EFIDP_MEDIA_HD) {
				fmt::print(stderr, "Entry {:04X} not Media Device Path HD. Abandoning.\n", went.bootnum_hint);
				continue;
			}

			if(cfg.verbose) {
				fmt::print("Entry {:04X}: ", went.bootnum_hint);
				klapki::context::detail::print_devpath(dp, efi_opt_len);
			}

			const efidp_data * file;
			switch(efidp_next_node(dp, &file)) {
				case -1:
					fmt::print(stderr, "Entry {:04X}: second (file) node: {}. Abandoning.\n", went.bootnum_hint, strerror(errno));
					continue;
				case 0:
					fmt::print(stderr, "Entry {:04X}: second (file) node doesn't exist. Abandoning.\n", went.bootnum_hint);
					continue;
				case 1: {
					if(file->type != EFIDP_MEDIA_TYPE && file->subtype != EFIDP_MEDIA_FILE) {
						fmt::print(stderr, "Entry {:04X} file not Media Device Path File. Abandoning.\n", went.bootnum_hint);
						continue;
					}

					const std::unique_ptr<char[], free_deleter> path_c{
					    reinterpret_cast<char *>(ucs2_to_utf8(reinterpret_cast<const char16_t *>(file->file.name), file->length))};
					std::string_view path{path_c.get()};
					if(const auto last_backslash = path.rfind('\\'); last_backslash != std::string::npos)
						kern.image_path = std::pair{path.substr(0, last_backslash + 1), path.substr(last_backslash + 1)};
					else {
						fmt::print(stderr, "Entry {:04X}: kernel image {}: no backslashes? Relating to root.\n", went.bootnum_hint, path);
						kern.image_path = std::pair{"\\", path};
					}
				} break;
				default:
					__builtin_unreachable();
			}

			const efidp_data * final_node;
			switch(efidp_next_node(file, &final_node)) {
				case -1:
					fmt::print(stderr, "Entry {:04X}: third (final) node: {}. Abandoning.\n", went.bootnum_hint, strerror(errno));
					continue;
				case 0:
					fmt::print(stderr, "Entry {:04X}: third (final) node doesn't exist. Abandoning.\n", went.bootnum_hint);
					continue;
				case 1: {
					if(final_node->type != EFIDP_END_TYPE || final_node->subtype != EFIDP_END_ENTIRE) {
						fmt::print(stderr, "Entry {:04X} final node not End Type End Entire. Abandoning.\n", went.bootnum_hint);
						continue;
					}
				} break;
				default:
					__builtin_unreachable();
			}

			if(cfg.verbose)
				fmt::print("Entry {:04X}: [\"{}\", \"{}\"]\n", went.bootnum_hint, kern.image_path.first, kern.image_path.second);

			kern.description = reinterpret_cast<const char *>(efi_loadopt_desc(efi_opt, efi_opt_len));
			if(cfg.verbose)
				fmt::print("Entry {:04X}: {}\n", went.bootnum_hint, kern.description);

			unsigned char * opt_data;
			size_t opt_data_len;
			efi_loadopt_optional_data(efi_opt, efi_opt_len, &opt_data, &opt_data_len);
			const std::unique_ptr<char[], free_deleter> opt_data_c{
			    reinterpret_cast<char *>(ucs2_to_utf8(reinterpret_cast<const char16_t *>(opt_data), opt_data_len))};
			detail::tokenise_cmdline(opt_data_c.get(), [&](auto && arg) {
				if(arg.substr(0, std::strlen("initrd=")) == "initrd=") {  // string_view::starts_with() is C++20
					if(kern.initrd_paths.size() == went.initrd_dirnames.size())
						fmt::print(stderr, "Entry {:04X}: added by cmdline: {}\n", went.bootnum_hint, arg);
					else {
						auto initrd_path = arg.substr(std::strlen("initrd="));
						if(const auto last_backslash = initrd_path.rfind('\\'); last_backslash != std::string::npos)
							kern.initrd_paths.emplace_back(initrd_path.substr(0, last_backslash + 1), initrd_path.substr(last_backslash + 1));
						else {
							fmt::print(stderr, "Entry {:04X}: initrd={}: no backslashes? Relating to root.\n", went.bootnum_hint, initrd_path);
							kern.initrd_paths.emplace_back("\\", initrd_path);
						}
					}
				}

				// This mangles ws but it's for informational purposes only — we get the freshest one during wisen() anyway
				kern.cmdline += arg;
				kern.cmdline += ' ';

				return true;
			});
			if(!kern.cmdline.empty())
				kern.cmdline.pop_back();  // extra space

			ret.our_kernels.emplace(went.bootnum_hint, std::move(kern));
		} else
			throw __func__;  // unreachable, by this point all wanted entries match up to boot entries

	return ret;
}
