// The MIT License (MIT)

// Copyright (c) 2020 наб <nabijaczleweli@nabijaczleweli.xyz>

// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


#pragma once


#include "ops.hpp"
#include <string>
#include <string_view>
#include <variant>
#include <vector>


namespace klapki {
	struct config {
		std::string host;
		constexpr std::string_view wisdom_root() const noexcept;
		bool verbose;
		bool commit;
		/// Tries /boot/efi, or /boot if that doesn't exist. Must be a mount-point.
		std::string_view esp;
		std::vector<ops::op_t> ops;

		std::variant<std::string_view, std::string> wisdom_root_raw;

		std::string news_efi_dir() const;

		static std::variant<config, std::string> read(const char ** argv);
	};
}


constexpr std::string_view klapki::config::wisdom_root() const noexcept {
	return std::visit([](auto && w) { return std::string_view{w}; }, this->wisdom_root_raw);
}

template <>
struct fmt::formatter<klapki::config> {
	constexpr auto parse(format_parse_context & ctx) { return ctx.begin(); }

	template <typename FormatContext>
	auto format(const klapki::config & cfg, FormatContext & ctx) {
		return format_to(ctx.out(), "config{{ host: \"{}\", wisdom: \"{}\", verbose: {}, commit: {}, ESP: {}, ops: [{}] }}",  //
		                 cfg.host, cfg.wisdom_root(), cfg.verbose, cfg.commit, cfg.esp, klapki::mops{cfg.ops});
	}
};
