// The MIT License (MIT)

// Copyright (c) 2020 наб <nabijaczleweli@nabijaczleweli.xyz>

// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


#include "config.hpp"
#include "context.hpp"
#include "context_detail.hpp"
#include "quickscope_wrapper.hpp"
#include <algorithm>
#include <dirent.h>
#include <fcntl.h>
#include <openssl/sha.h>
#include <set>
#include <strings.h>
#include <sys/mman.h>
#include <sys/sendfile.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>


#define TRY(...)                                       \
	({                                                   \
		auto ret = __VA_ARGS__;                            \
		if(auto err = std::get_if<std::string>(&ret); err) \
			return std::move(*err);                          \
		std::move(std::get<0>(ret));                       \
	})

#define TRY_OPT(...)              \
	if(auto err = __VA_ARGS__; err) \
		return err;


using sha_t = std::uint8_t[20];


namespace {
	struct caseless {
		using is_transparent = void;

		bool operator()(const std::string & l, const std::string & r) const noexcept { return strcasecmp(l.c_str(), r.c_str()) < 0; }
		bool operator()(const std::string & l, const char * r) const noexcept { return strcasecmp(l.c_str(), r) < 0; }
		bool operator()(const std::string_view & l, const std::string_view & r) const noexcept { return strcasecmp(l.data(), r.data()) < 0; }
		bool operator()(const std::string_view & l, const char * r) const noexcept { return strcasecmp(l.data(), r) < 0; }
	};

	/// https://stackoverflow.com/a/2180157/2851815 says FreeBSD and Darwin have fcopyfile(3),
	/// but the only reference I could find was (a) copies of that answer and (b) Apple documentation, which wasn't helpful.
	///
	/// Plus, I've no idea if libefivar even works on Berkeley distros; it's got separate linux*.c implementations, but.
	///
	/// Just use sendfile(2) here and potentially ifdef for non-Linux later.
	std::variant<bool, std::string> copy_file(int srcdir, int destdir, const char * basename, sha_t cursha, bool verbose) {
		auto srcfd = openat(srcdir, basename, O_RDONLY);
		if(srcfd < 0)
			return fmt::format("Couldn't open {} for reading: {}\n", basename, strerror(errno));
		klapki::quickscope_wrapper srcfd_deleter{[&] { close(srcfd); }};

		struct stat src_sb;
		if(fstat(srcfd, &src_sb) < 0)
			return fmt::format("Couldn't stat() source {}: {}\n", basename, strerror(errno));

		sha_t insha{};
		{
			auto map = mmap(nullptr, src_sb.st_size, PROT_READ, MAP_PRIVATE, srcfd, 0);
			if(!map)
				return fmt::format("Couldn't mmap() source {}: {}\n", basename, strerror(errno));
			klapki::quickscope_wrapper map_deleter{[&] {
				if(munmap(map, src_sb.st_size))
					fmt::print(stderr, "munmap {}: {}\n", basename, strerror(errno));
			}};

			SHA1(reinterpret_cast<const unsigned char *>(map), src_sb.st_size, insha);
		}
		if(!std::memcmp(cursha, insha, sizeof(sha_t))) {
			if(verbose)
				fmt::print("{} unchanged ({})\n", basename, klapki::context::detail::sha_f{cursha});

			struct stat dest_sb;  // Still copy if missning
			if(fstatat(destdir, basename, &dest_sb, 0) < 0) {
				if(errno != ENOENT)
					return fmt::format("Couldn't stat() source {}: {}\n", basename, strerror(errno));
			} else
				return false;
		}

		if(verbose)
			fmt::print("{} changed ({} -> {})\n", basename, klapki::context::detail::sha_f{cursha}, klapki::context::detail::sha_f{insha});
		std::memcpy(cursha, insha, sizeof(sha_t));

		auto destfd = openat(destdir, basename, O_WRONLY | O_CREAT | O_TRUNC, 0644);
		if(destfd < 0)
			return fmt::format("Couldn't open {} for writing: {}\n", basename, strerror(errno));
		klapki::quickscope_wrapper destfd_deleter{[&] {
			if(close(destfd))
				fmt::print(stderr, "Warning: destination {} errored on close: {}", basename, strerror(errno));
		}};

		off_t offset = 0;
		for(ssize_t res = 0; res < src_sb.st_size; res = sendfile(destfd, srcfd, &offset, src_sb.st_size)) {
			if(res < 0)
				return fmt::format("Couldn't copy to {} (offset {}): {}\n", basename, offset, strerror(errno));

			src_sb.st_size -= res;
		}

		return true;
	}
}


std::optional<std::string> klapki::context::context::commit(const config & cfg, state::state & state) const {
	auto esp_fd = open(cfg.esp.data(), O_RDONLY | O_DIRECTORY | O_PATH);
	if(esp_fd < 0)
		return fmt::format("Couldn't open ESP ({}): {}\n", cfg.esp, strerror(errno));
	quickscope_wrapper esp_fd_deleter{[&] { close(esp_fd); }};

	std::map<std::string_view, int, caseless> esp_dirs;
	quickscope_wrapper esp_dirs_deleter{[&] {
		for(auto && [dir, fd] : esp_dirs)
			close(fd);
	}};
	std::set<std::string> deleted_dir_names;
	auto adddir = [&](auto && ddir, auto && cbk) -> std::optional<std::string> {
		if(esp_dirs.find(ddir) == std::end(esp_dirs)) {
			auto dir = ddir;
			std::replace(std::begin(dir), std::end(dir), '\\', '/');
			dir.erase(std::remove_if(std::begin(dir), std::end(dir),
			                         [prev = '\0'](auto c) mutable {
				                         if(prev == '/' && c == '/')
					                         return true;
				                         else {
					                         prev = c;
					                         return false;
				                         }
			                         }),
			          std::end(dir));
			if(dir.back() == '/')
				dir.pop_back();
			dir.erase(0, dir.find_first_not_of('/'));

			for(auto slash_idx = dir.find('/'); slash_idx != std::string::npos; slash_idx = dir.find('/', slash_idx + 1)) {
				dir[slash_idx] = '\0';
				if(mkdirat(esp_fd, dir.c_str(), 0755) < 0 && errno != EEXIST)
					return fmt::format("Couldn't create {} under ESP ({}): {}\n", dir.c_str(), cfg.esp, strerror(errno));
				dir[slash_idx] = '/';
			}
			if(mkdirat(esp_fd, dir.c_str(), 0755) < 0 && errno != EEXIST)
				return fmt::format("Couldn't create {} under ESP ({}): {}\n", dir, cfg.esp, strerror(errno));

			auto fd = openat(esp_fd, dir.c_str(), O_RDONLY | O_DIRECTORY | O_PATH);
			if(fd < 0)
				return fmt::format("Couldn't open {} under ESP ({}): {}\n", dir, cfg.esp, strerror(errno));
			esp_dirs.emplace(ddir, fd);
			cbk(std::move(dir));
		}

		return {};
	};
	for(auto && kern : this->our_kernels)
		TRY_OPT(adddir(kern.second.image_path.first, [](auto &&) {}));
	for(auto && [ddir, _] : this->deleted_files)
		TRY_OPT(adddir(ddir, [&](auto && dname) { deleted_dir_names.emplace(std::move(dname)); }));


	std::map<detail::bad_cow, std::pair<DIR *, std::set<std::string, caseless>>> source_dirs;
	quickscope_wrapper dir_fds_deleter{[&] {
		for(auto && [path, dir_ents] : source_dirs)
			if(dir_ents.first)
				closedir(dir_ents.first);  // No error checking, just directories
	}};

	for(auto && skern : state.statecfg.wanted_entries) {
		source_dirs[detail::bad_cow{skern.kernel_dirname}].first = nullptr;
		for(auto && [initrd_dirname, _] : skern.initrd_dirnames)
			if(initrd_dirname)
				source_dirs[detail::bad_cow{*initrd_dirname}].first = nullptr;
	}

	for(auto && [path, dir_ents] : source_dirs) {
		dir_ents.first = opendir(path.get().data());
		if(!dir_ents.first)
			return fmt::format("Couldn't open {}: {}\n", path.get(), strerror(errno));

		while(const auto ent = readdir(dir_ents.first))
			dir_ents.second.emplace(ent->d_name);
	}


	std::map<std::pair<int, std::string>, const std::uint8_t *> already_copied;
	auto do_copy = [&](auto bootnum, auto && srcdir, auto && destdir, const auto * basename, auto cursha) -> std::optional<std::string> {
		auto destfd = esp_dirs.find(destdir);
		if(destfd == std::end(esp_dirs))
			return fmt::format("ESP dir {} not in cache?", destfd->first);

		auto srcfd = source_dirs.find(detail::bad_cow{srcdir});
		if(srcfd == std::end(source_dirs))
			return fmt::format("Source dir {} not in cache?", srcdir);

		if(const auto entry = srcfd->second.second.find(*basename); entry != srcfd->second.second.end())
			basename = &*entry;

		if(auto ac = already_copied.find(std::pair{destfd->second, *basename}); ac != std::end(already_copied)) {
			if(cfg.verbose)
				fmt::print("Entry {:04X}: already copied {} from {} to {}\n", bootnum, *basename, srcdir, destfd->first);
			std::memcpy(cursha, ac->second, sizeof(sha_t));
			return {};
		}

		if(cfg.verbose)
			fmt::print("Entry {:04X}: copying {} from {} to {}\n", bootnum, *basename, srcdir, destfd->first);
		if(TRY(copy_file(dirfd(srcfd->second.first), destfd->second, basename->c_str(), cursha, cfg.verbose)))
			fmt::print("Entry {:04X}: copied {} from {} to {}\n", bootnum, *basename, srcdir, destfd->first);
		already_copied.emplace(std::pair{destfd->second, *basename}, cursha);
		return {};
	};

	for(auto && [bootnum, kern] : this->our_kernels) {
		auto skern = std::find_if(std::begin(state.statecfg.wanted_entries), std::end(state.statecfg.wanted_entries),
		                          [bn = bootnum](auto && skern) { return skern.bootnum_hint == bn; });
		if(skern == std::end(state.statecfg.wanted_entries))
			throw __func__;

		TRY_OPT(do_copy(bootnum, skern->kernel_dirname, kern.image_path.first, &kern.image_path.second, skern->kernel_image_sha));

		if(skern->initrd_dirnames.size() != kern.initrd_paths.size())
			throw __func__;

		std::string_view sprev = skern->kernel_dirname;
		std::string_view dprev = kern.image_path.first;
		for(std::size_t i = 0; i < skern->initrd_dirnames.size(); ++i) {
			auto && [didir, dibase] = kern.initrd_paths[i];
			auto && [sidir, ssha]   = skern->initrd_dirnames[i];

			if(sidir)
				sprev = *sidir;
			if(didir)
				dprev = *didir;
			TRY_OPT(do_copy(bootnum, sprev, dprev, &dibase, &ssha[0]));
		}
	}


	for(auto && [ddir, dfile] : this->deleted_files) {
		auto destfd = esp_dirs.find(ddir);
		if(destfd == std::end(esp_dirs))
			return fmt::format("ESP dir {} not in cache?", ddir);

		if(already_copied.find(std::pair{destfd->second, dfile}) != std::end(already_copied)) {
			if(cfg.verbose)
				fmt::print("Not removing {} from {} after having copied it there\n", dfile, ddir);
			continue;
		}

		fmt::print("Removing {} from {}\n", dfile, ddir);
		if(unlinkat(destfd->second, dfile.c_str(), 0) < 0 && unlinkat(destfd->second, dfile.c_str(), AT_REMOVEDIR) < 0)
			return fmt::format("Removing {} from {}: {}", dfile, ddir, strerror(errno));
	}

	for(auto && ddir : deleted_dir_names) {
		if(cfg.verbose)
			fmt::print("Cleaning up {} from {}\n", ddir, cfg.esp);
		if(unlinkat(esp_fd, ddir.c_str(), AT_REMOVEDIR) < 0)
			if(cfg.verbose)
				fmt::print("Removing {} from {}: {}\n", ddir, cfg.esp, strerror(errno));
	}


	return {};
}
