// The MIT License (MIT)

// Copyright (c) 2020 наб <nabijaczleweli@nabijaczleweli.xyz>

// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


#include "config.hpp"
#include "context.hpp"
#include "state.hpp"
#include "util.hpp"
#include <fmt/format.h>


#define TRY(ec, ...)                                           \
	({                                                           \
		auto ret = __VA_ARGS__;                                    \
		if(auto errmsg = std::get_if<std::string>(&ret); errmsg) { \
			fmt::print(stderr, "{}\n", *errmsg);                     \
			return ec;                                               \
		}                                                          \
		std::move(std::get<0>(ret));                               \
	})

#define TRY_OPT(ec, pref, ...)          \
	if(auto err = __VA_ARGS__; err) {     \
		fmt::print("{}: {}\n", pref, *err); \
		return ec;                          \
	}


namespace klapki {
	static int main(const char ** argv) {
		auto cfg = TRY(1, config::read(argv));
		if(cfg.verbose)
			fmt::print("{}\n", cfg);


		const auto input_state = TRY(2, state::state::load(cfg.host));


		auto output_state = TRY(3, context::resolve_state_context(cfg, input_state));


		auto context = context::context::derive(cfg, output_state);


		for(auto && op : cfg.ops) {
			if(cfg.verbose)
				fmt::print("Running {}\n", klapki::op{op});

			TRY_OPT(4, klapki::op{op}, op::execute(op, cfg, output_state, context));
		}


		TRY_OPT(5, "context.propagate()", context.propagate(cfg, output_state));
		if(cfg.verbose) {
			fmt::print("Propagated:");
			op::execute(klapki::ops::dump{}, cfg, output_state, context);
		}

		TRY_OPT(6, "context.age()", context.age(cfg, output_state));
		if(cfg.verbose) {
			fmt::print("Aged:");
			op::execute(klapki::ops::dump{}, cfg, output_state, context);
		}

		TRY_OPT(7, "context.wisen()", context.wisen(cfg, output_state));
		if(cfg.verbose) {
			fmt::print("Wisened:");
			op::execute(klapki::ops::dump{}, cfg, output_state, context);
		}

		TRY_OPT(8, "context.save()", context.save(cfg, output_state));
		if(cfg.verbose) {
			fmt::print("Saved:");
			op::execute(klapki::ops::dump{}, cfg, output_state, context);
		}

		if(cfg.commit) {  // Both const, context only updates SHAs which we don't print anyway
			TRY_OPT(9, "context.commit()", context.commit(cfg, output_state));
			TRY_OPT(10, "output_state.commit()", output_state.commit(cfg.host, input_state));
		}

		return 0;
	}
}

int main(int, const char ** argv) { return klapki::main(argv); }
