// The MIT License (MIT)

// Copyright (c) 2020 наб <nabijaczleweli@nabijaczleweli.xyz>

// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


#include "config.hpp"
#include "context.hpp"
#include "ops.hpp"
#include <fmt/format.h>


namespace {
	static void slash_path(std::pair<std::string_view, std::string_view> & which, const char * whose) {
		std::string_view path{whose};
		if(auto last_slash = path.rfind('/'); last_slash != std::string::npos) {
			which.first = path.substr(0, last_slash);  // drop trailing slash here
			path        = path.substr(last_slash + 1);
		} else
			which.first = ".";  // age() will resolve this to abspath
		which.second = path;
	}
}


std::optional<std::string> klapki::ops::execute(const klapki::ops::dump &, const klapki::config &, klapki::state::state & state, context::context & context) {
	fmt::print("\n"
	           "Boot order: {}\n"
	           "{} boot entries\n"
	           "Desired boot position: {}\n"
	           "Boot variants: ",
	           state.order, state.entries.size(), state.statecfg.boot_position);

	if(state.statecfg.variants.empty())
		fmt::print("(none)");
	else {
		bool first = true;
		for(auto && el : state.statecfg.variants) {
			if(!first)
				fmt::print(", ");
			else
				first = false;

			fmt::print("{}", el);
		}
	}
	fmt::print("\n"
	           "Wanted entries: [");

	bool first = true;
	for(auto && el : state.statecfg.wanted_entries) {
		if(first) {
			fmt::print("\n");
			first = false;
		}

		fmt::print("  {}\n", el);
	}

	fmt::print("]\n"
	           "\n"
	           "{}\n"
	           "\n",
	           context);

	return {};
}

std::optional<std::string> klapki::ops::execute(const klapki::ops::bootpos & bp, const klapki::config &, klapki::state::state & state, context::context &) {
	state.statecfg.boot_position = bp.position;
	return {};
}

std::optional<std::string> klapki::ops::execute(const klapki::ops::addkernel & ak, const klapki::config &, klapki::state::state & state,
                                                context::context & context) {
	if(std::find_if(std::begin(state.statecfg.wanted_entries), std::end(state.statecfg.wanted_entries),
	                [&](auto && went) { return went.version == ak.version; }) != std::end(state.statecfg.wanted_entries)) {
		fmt::print(stderr, "addkernel: kernel version {} already known\n", ak.version);
		return {};
	}

	klapki::context::fresh_kernel kern{ak.version, {}, {}};

	slash_path(kern.image, ak.image);
	context.fresh_kernels.reserve(ak.initrds.size());
	for(auto && initrd : ak.initrds) {
		kern.initrds.emplace_back();
		slash_path(kern.initrds.back(), initrd);
	}

	context.fresh_kernels.emplace_back(std::move(kern));
	return {};
}

std::optional<std::string> klapki::ops::execute(const klapki::ops::delkernel & dk, const klapki::config &, klapki::state::state & state,
                                                context::context & context) {
	context.purge_allocations(state, [&](auto && skern) { return skern.version == dk.version; });

	context.fresh_kernels.erase(
	    std::remove_if(std::begin(context.fresh_kernels), std::end(context.fresh_kernels), [&](auto && fkern) { return fkern.version == dk.version; }),
	    std::end(context.fresh_kernels));

	return {};
}

std::optional<std::string> klapki::ops::execute(const klapki::ops::addvariant & av, const klapki::config &, klapki::state::state & state, context::context &) {
	if(av.variant[0] == '\0' ||
	   std::find(std::begin(state.statecfg.variants), std::end(state.statecfg.variants), av.variant) != std::end(state.statecfg.variants))
		fmt::print(stderr, "addvariant: boot variant {} already known\n", av.variant[0] == '\0' ? "(default)" : av.variant);
	else
		state.statecfg.variants.push_back(av.variant);

	return {};
}

std::optional<std::string> klapki::ops::execute(const klapki::ops::delvariant & dv, const klapki::config &, klapki::state::state & state, context::context &) {
	if(auto itr = std::find(std::begin(state.statecfg.variants), std::end(state.statecfg.variants), dv.variant); itr != std::end(state.statecfg.variants))
		state.statecfg.variants.erase(itr);
	else
		fmt::print(stderr, "delvariant: boot variant {} not known\n", dv.variant);

	return {};
}
