// The MIT License (MIT)

// Copyright (c) 2020 наб <nabijaczleweli@nabijaczleweli.xyz>

// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


#include "config.hpp"
#include "util.hpp"
#include <cstdlib>
#include <fmt/format.h>
#include <fstream>
#include <limits.h>
#include <sys/stat.h>
#include <sys/utsname.h>
#include <unistd.h>
extern "C" {
#include <efivar/efivar.h>
}


#define TRY(...)                                       \
	({                                                   \
		auto ret = __VA_ARGS__;                            \
		if(auto err = std::get_if<std::string>(&ret); err) \
			return std::move(*err);                          \
		std::move(std::get<0>(ret));                       \
	})


namespace {
	std::string readline(const char * from) {
		std::ifstream in(from);
		std::string ret;
		std::getline(in, ret);
		return ret;
	}

	std::string get_host() {
		if(auto host = std::getenv("KLAPKI_HOST"))
			return host;
		else {
			auto mid = readline("/etc/machine-id");
			if(!mid.empty())
				return mid;

			char hostname[HOST_NAME_MAX + 1];
			gethostname(hostname, sizeof(hostname) / sizeof(*hostname));
			return hostname;
		}
	}

	std::variant<std::string_view, std::string> get_wisom_root() {
		if(auto host = std::getenv("KLAPKI_WISDOM")) {
			std::string ret{host};
			if(!ret.empty() && ret.back() != '/')
				ret += '/';
			return ret;
		} else
			return std::string_view{"/etc/klapki/"};
	}

	std::variant<std::string_view, std::string> find_esp() {
		const char * esp = "/boot/efi";

		struct stat boot;
		if(stat("/boot", &boot) < 0)
			return fmt::format("find_esp(): stat(\"/boot\"): {}", std::strerror(errno));

		struct stat boot_efi;
		if(stat("/boot/efi", &boot_efi) < 0) {
			if(errno != ENOENT)
				return fmt::format("find_esp(): stat(\"/boot/efi\"): {}", std::strerror(errno));
			else {
				esp      = "/boot";
				boot_efi = boot;
				if(stat("/", &boot) < 0)
					return fmt::format("find_esp(): stat(\"/\"): {}", std::strerror(errno));
			}
		}

		if(boot.st_dev == boot_efi.st_dev)
			return fmt::format("find_esp(): {}: not a mount point?", esp);

		return std::string_view{esp};
	}
}


std::string klapki::config::news_efi_dir() const {
	std::string broot;
	std::string_view vroot = "klapki";
	if(auto root = std::getenv("KLAPKI_EFI_ROOT")) {
		while(*root == '\\' || *root == '/')
			++root;
		broot = root;
		std::replace(std::begin(broot), std::end(broot), '/', '\\');
		while(!broot.empty() && broot.back() == '\\')
			broot.pop_back();
		vroot = broot;
	}
	return fmt::format("\\{}{}{}\\", vroot, vroot.empty() ? "" : "\\", this->host);
}


std::variant<klapki::config, std::string> klapki::config::read(const char ** argv) {
	const auto argv0 = argv[0];
	++argv;

	bool verbose  = false;
	bool verybose = false;
	bool commit   = true;
	for(; *argv && argv[0][0] == '-'; ++argv) {
		for(auto opts = argv[0] + 1; *opts; ++opts)
			switch(opts[0]) {
				case 'v':
					verbose = true;
					break;

				case 'V':
					verybose = true;
					break;

				case 'E':
					efi_set_verbose(efi_get_verbose() + 1, nullptr);
					break;

				case 'n':
					commit = false;
					break;

				case 'h':
					// Remember to sync to klapki(8)
					return fmt::format("klapki {}\n"
					                   "Usage: {} [-nvVEh]… [op [arg…]]…\n"
					                   "\n"
					                   "Flags:\n"
					                   "  -n\tDon't commit\n"
					                   "  -v\tVerbose operation\n"
					                   "  -V\tAdd {{dump}}s around all ops\n"
					                   "  -E\tIncrease libefivar verbosity level\n"
					                   "  -h\tShow this help\n"
					                   "\n"
					                   "Environment:\n"
					                   "  KLAPKI_HOST=    \tReplaces contents of /etc/machine-id or hostname\n"
					                   "  KLAPKI_WISDOM=  \tReplaces /etc/klapki\n"
					                   "  KLAPKI_EFI_ROOT=\tReplaces \\klapki\n"
					                   "\n"
					                   "Recognised ops:\n"
					                   "\tdump\n"
					                   "\tbootpos <position>\n"
					                   "\taddkernel <version> <image> [initrd…] <\"\">\n"
					                   "\tdelkernel <version>\n"
					                   "\taddvariant <variant>\n"
					                   "\tdelvariant <variant>",
					                   KLAPKI_VERSION, argv0);

				default:
					return fmt::format("{}: unknown flag {}", argv0, opts[0]);
			}
	}

	std::vector<ops::op_t> ops;
	if(verybose)
		ops.emplace_back(ops::dump{});
	while(*argv) {
		auto pop = op::from_cmdline(argv0, argv);
		if(auto err = std::get_if<std::string>(&pop))
			return std::move(*err);
		else {
			ops.emplace_back(std::move(std::get<ops::op_t>(pop)));
			if(verybose)
				ops.emplace_back(ops::dump{});
		}
	}

	return config{get_host(), verbose, commit, TRY(find_esp()), std::move(ops), get_wisom_root()};
}
