// The MIT License (MIT)

// Copyright (c) 2020 наб <nabijaczleweli@nabijaczleweli.xyz>

// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


#include "config.hpp"
#include "context.hpp"
#include "context_detail.hpp"
#include "quickscope_wrapper.hpp"
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>


#define TRY(...)                                       \
	({                                                   \
		auto ret = __VA_ARGS__;                            \
		if(auto err = std::get_if<std::string>(&ret); err) \
			return std::move(*err);                          \
		std::move(std::get<0>(ret));                       \
	})


namespace {
	struct wisdom {
		std::string data;
	};

	std::variant<wisdom, std::string> gain_wisdom(const klapki::config & cfg, const char * from, const char * version, const char * variant) {
		int comm[2];
		if(pipe2(comm, O_CLOEXEC) < 0)
			return fmt::format("opening {} {} {} file: {}", from, version, variant, std::strerror(errno));

		switch(auto pid = fork()) {
			case -1:
				close(comm[0]);
				close(comm[1]);
				return fmt::format("forking for {} {} {}: {}", from, version, variant, std::strerror(errno));

			case 0: {  // child
				if(dup2(comm[1], 1) < 0)
					_exit(0x6B);

				const auto hook = fmt::format("{}{}", cfg.wisdom_root(), from);
				execl(hook.c_str(), from, version, variant, (char *)nullptr);
				fmt::print(stderr, "exec {} {} {}: {}\n", hook, version, variant, std::strerror(errno));
				_exit(0x6B);
			}

			default: {  // parent
				close(comm[1]);
				klapki::quickscope_wrapper cmdline_fd_del{[&] { close(comm[0]); }};

				wisdom ret;
				char buf[64 * 1024];
				for(ssize_t rd = -1; rd;) {
					while((rd = read(comm[0], buf, sizeof(buf))) == -1 && errno == EINTR)
						;
					if(rd < 0)
						return fmt::format("{} read failed with {}", from, std::strerror(errno));
					ret.data.append(buf, rd);
				}

				int result;
				waitpid(pid, &result, 0);
				if(WIFEXITED(result) && WEXITSTATUS(result) != 0)
					return fmt::format("{} child returned {}", from, WEXITSTATUS(result));
				else if(WIFSIGNALED(result))
					return fmt::format("{} child killed by signal {} ({})", from, WTERMSIG(result), strsignal(WTERMSIG(result)));

				if(auto idx = ret.data.find('\0'); idx != std::string::npos)  // end at first NUL, if any; prevent unexpected behaviour by behaving like a C string
					ret.data.erase(idx);

				// Character set stolen from isspace()
				ret.data.erase(ret.data.find_last_not_of(" \f\n\r\t\v") + 1);
				ret.data.erase(0, ret.data.find_first_not_of(" \f\n\r\t\v"));

				std::replace(std::begin(ret.data), std::end(ret.data), '\n', ' ');

				return ret;
			}
		}
	}

	void validate_cmdline(const std::string & cmdline) {
		klapki::context::detail::tokenise_cmdline(cmdline, [&](auto && arg) {
			if(arg.substr(0, std::strlen("initrd=")) == "initrd=") {  // string::starts_with() is C++20
				fmt::print(stderr, "Stray {} in cmdline, things might not work as planned\n", arg);
				return false;
			} else
				return true;
		});
	}
}

std::optional<std::string> klapki::context::context::wisen(const config & cfg, state::state & state) {
	for(auto && [bootnum, kern] : this->our_kernels) {
		auto skern = std::find_if(std::begin(state.statecfg.wanted_entries), std::end(state.statecfg.wanted_entries),
		                          [bn = bootnum](auto && skern) { return skern.bootnum_hint == bn; });
		if(skern == std::end(state.statecfg.wanted_entries))
			throw __func__;

		kern.description = std::move(TRY(gain_wisdom(cfg, "description", skern->version.c_str(), skern->variant.c_str())).data);
		kern.cmdline     = std::move(TRY(gain_wisdom(cfg, "cmdline", skern->version.c_str(), skern->variant.c_str())).data);
		validate_cmdline(kern.cmdline);
	}

	return {};
}
