// The MIT License (MIT)

// Copyright (c) 2020 наб <nabijaczleweli@nabijaczleweli.xyz>

// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


#pragma once


#include "state.hpp"
#include <algorithm>
#include <fmt/format.h>
#include <map>
#include <set>
#include <string_view>
#include <variant>
#include <vector>
extern "C" {
#include <efivar/efiboot.h>
#include <efivar/efivar-dp.h>
}

namespace klapki {
	struct config;
}


namespace klapki::context {
	struct our_kernel {
		std::string description;
		std::string cmdline;

		std::pair<std::string, std::string> image_path;                              // path in ESP, basename
		std::vector<std::pair<state::nonbase_dirname_t, std::string>> initrd_paths;  // path in ESP, basename
	};

	struct fresh_kernel {
		std::string_view version;
		std::pair<std::string_view, std::string_view> image;
		std::vector<std::pair<std::string_view, std::string_view>> initrds;
	};

	struct context {
		std::map<std::uint16_t, our_kernel> our_kernels;
		std::vector<fresh_kernel> fresh_kernels;

		std::set<std::pair<std::string, std::string>> deleted_files;  // path in ESP, basename

		std::optional<std::string> allocate_kernel_variant(const config & cfg, state::state & state, std::string version, std::string var, std::string image_dir,
		                                                   state::sha_t & image_sha, std::vector<std::pair<state::nonbase_dirname_t, state::shaa_t>> initrd_dirs,
		                                                   std::string image_basename, std::vector<std::pair<state::nonbase_dirname_t, std::string>> initrd_paths);
		template <class F>
		void purge_allocations(state::state & state, F && pred);
		void purge_allocations_impl(state::state & state, const state::stated_config_entry & skern);

		std::optional<std::string> propagate(const config & cfg, state::state & state);
		std::optional<std::string> age(const config & cfg, state::state & state);
		std::optional<std::string> wisen(const config & cfg, state::state & state);
		std::optional<std::string> save(const config & cfg, state::state & state);
		std::optional<std::string> commit(const config & cfg, state::state & state) const;  // Only updates file SHAs in state

		static context derive(const config & cfg, state::state & input_state);
	};

	std::variant<state::state, std::string> resolve_state_context(const config & cfg, const state::state & input_state);
}


template <class F>
void klapki::context::context::purge_allocations(state::state & state, F && pred) {
	state.statecfg.wanted_entries.erase(std::remove_if(std::begin(state.statecfg.wanted_entries), std::end(state.statecfg.wanted_entries),
	                                                   [&](auto && skern) {
		                                                   if(!pred(skern))
			                                                   return false;

		                                                   this->purge_allocations_impl(state, skern);
		                                                   return true;
	                                                   }),
	                                    std::end(state.statecfg.wanted_entries));
}


template <>
struct fmt::formatter<klapki::context::context> {
	constexpr auto parse(format_parse_context & ctx) { return ctx.begin(); }

	template <typename FormatContext>
	auto format(const klapki::context::context & context, FormatContext & ctx) {
		auto out = ctx.out();

		out = format_to(out, "{} of our kernels: [", context.our_kernels.size());

		bool first = true;
		for(auto && el : context.our_kernels) {
			if(first) {
				*out++ = '\n';
				first  = false;
			}

			out = format_to(out, "  {:04X}: {}\n", el.first, el.second);
		}


		out = format_to(out, "]\n"
		                     "Fresh kernels: [");

		first = true;
		for(auto && el : context.fresh_kernels) {
			if(first) {
				*out++ = '\n';
				first  = false;
			}

			out = format_to(out, "  {}\n", el);
		}

		out = format_to(out, "]\n"
		                     "Deleted files: [");

		first = true;
		for(auto && el : context.deleted_files) {
			if(first) {
				*out++ = '\n';
				first  = false;
			}

			out = format_to(out, "  (\"{}\", \"{}\")\n", el.first, el.second);
		}

		out = format_to(out, "]");

		return out;
	}
};

template <>
struct fmt::formatter<klapki::context::our_kernel> {
	constexpr auto parse(format_parse_context & ctx) { return ctx.begin(); }

	template <typename FormatContext>
	auto format(const klapki::context::our_kernel & kern, FormatContext & ctx) {
		auto out = ctx.out();

		out = format_to(out, "{{ \"{}\", \"{}\", (\"{}\", \"{}\"), [", kern.description, kern.cmdline, kern.image_path.first, kern.image_path.second);

		bool first = true;
		for(auto && el : kern.initrd_paths) {
			if(!first)
				out = format_to(out, ", ");
			else
				first = false;

			*out++ = '(';
			if(el.first)
				*out++ = '\"';
			out = format_to(out, "{}", el.first ? std::string_view{*el.first} : "←");
			if(el.first)
				*out++ = '\"';
			out = format_to(out, ", \"{}\")", el.second);
		}

		out = format_to(out, "] }}");

		return out;
	}
};

template <>
struct fmt::formatter<klapki::context::fresh_kernel> {
	constexpr auto parse(format_parse_context & ctx) { return ctx.begin(); }

	template <typename FormatContext>
	auto format(const klapki::context::fresh_kernel & kern, FormatContext & ctx) {
		auto out = ctx.out();

		out = format_to(out, "{{ \"{}\", (\"{}\", \"{}\"), [", kern.version, kern.image.first, kern.image.second);

		bool first = true;
		for(auto && el : kern.initrds) {
			if(!first)
				out = format_to(out, ", ");
			else
				first = false;

			out = format_to(out, "(\"{}\", \"{}\")", el.first, el.second);
		}

		out = format_to(out, "] }}");

		return out;
	}
};
