// The MIT License (MIT)

// Copyright (c) 2020 наб <nabijaczleweli@nabijaczleweli.xyz>

// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


#pragma once


#include <fmt/format.h>
#include <numeric>
#include <string_view>
extern "C" {
#include <efivar/efiboot.h>
}


namespace klapki::context::detail {
	template <class F>
	void tokenise_cmdline(std::string_view chunk, F && func) {
		// Character set stolen from isspace()
		for(auto cur = chunk.find_first_not_of(" \f\n\r\t\v"); cur != std::string::npos; cur = chunk.find_first_not_of(" \f\n\r\t\v")) {
			chunk = chunk.substr(cur);

			auto next = chunk.find_first_of(" \f\n\r\t\v");
			if(!func(chunk.substr(0, next)))
				break;

			if(next == std::string::npos)
				break;
			chunk = chunk.substr(next);
		}
	}

	struct bad_cow {
		std::variant<std::string_view, std::string> data;

		constexpr std::string_view get() const noexcept {
			return std::visit([](auto && h) { return std::string_view{h}; }, this->data);
		}

		template <class O>
		constexpr bool operator==(const O & other) const noexcept {
			return this->get() == other;
		}
		constexpr bool operator==(const bad_cow & other) const noexcept { return this->get() == other.get(); }

		constexpr bool operator<(const bad_cow & other) const noexcept { return this->get() < other.get(); }
	};

	struct sha_f {
		const std::uint8_t * sha;
	};

	void print_devpath(const efidp_data * dp, ssize_t dp_len);
}


template <>
struct fmt::formatter<klapki::context::detail::sha_f> {
	constexpr auto parse(format_parse_context & ctx) { return ctx.begin(); }

	template <typename FormatContext>
	auto format(klapki::context::detail::sha_f s, FormatContext & ctx) {
		using sha_t = std::uint8_t[20];
		return std::accumulate(s.sha, s.sha + sizeof(sha_t), ctx.out(), [](auto && out, auto b) { return format_to(out, "{:02X}", b); });
	}
};
