// The MIT License (MIT)

// Copyright (c) 2020 наб <nabijaczleweli@nabijaczleweli.xyz>

// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


#pragma once


#include "state.hpp"
#include "util.hpp"
#include <cstdint>
#include <fmt/format.h>
#include <optional>
#include <string>
#include <variant>
#include <vector>

namespace klapki {
	struct config;
}
namespace klapki::context {
	struct context;
}


namespace klapki::ops {
	struct dump {};
	struct bootpos {
		std::uint16_t position;
	};
	struct addkernel {
		const char * version;
		const char * image;
		std::vector<const char *> initrds;
	};
	struct delkernel {
		const char * version;
	};
	struct addvariant {
		const char * variant;
	};
	struct delvariant {
		const char * variant;
	};

	using op_t = std::variant<dump, bootpos, addkernel, delkernel, addvariant, delvariant>;

	std::optional<std::string> execute(const dump & d, const config & cfg, state::state & state, context::context & context);
	std::optional<std::string> execute(const bootpos & bp, const config & cfg, state::state & state, context::context & context);
	std::optional<std::string> execute(const addkernel & ak, const config & cfg, state::state & state, context::context & context);
	std::optional<std::string> execute(const delkernel & dk, const config & cfg, state::state & state, context::context & context);
	std::optional<std::string> execute(const addvariant & av, const config & cfg, state::state & state, context::context & context);
	std::optional<std::string> execute(const delvariant & dv, const config & cfg, state::state & state, context::context & context);
}


namespace klapki {
	namespace detail {
		struct initrds_fmt {
			const std::vector<const char *> & initrds;
		};
	}

	struct op {
		const ops::op_t & op;

		static std::variant<ops::op_t, std::string> from_cmdline(const char * argv0, const char **& argv);
		static std::optional<std::string> execute(const ops::op_t & op, const config & cfg, state::state & state, context::context & context);
	};
	struct mops {
		const std::vector<ops::op_t> & ops;
	};
}


template <>
struct fmt::formatter<klapki::op> {
	constexpr auto parse(format_parse_context & ctx) { return ctx.begin(); }

	template <typename FormatContext>
	auto format(const klapki::op & op, FormatContext & ctx) {
		return std::visit(klapki::overload{
		                      [&](const klapki::ops::dump &) { return format_to(ctx.out(), "{{dump}}"); },
		                      [&](const klapki::ops::bootpos & bp) { return format_to(ctx.out(), "{{bootpos {}}}", bp.position); },
		                      [&](const klapki::ops::addkernel & ak) {
			                      return format_to(ctx.out(), "{{addkernel {} {} {}}}", ak.version, ak.image, klapki::detail::initrds_fmt{ak.initrds});
		                      },
		                      [&](const klapki::ops::delkernel & dk) { return format_to(ctx.out(), "{{delkernel {}}}", dk.version); },
		                      [&](const klapki::ops::addvariant & av) { return format_to(ctx.out(), "{{addvariant {}}}", av.variant); },
		                      [&](const klapki::ops::delvariant & dv) { return format_to(ctx.out(), "{{delvariant {}}}", dv.variant); },
		                  },
		                  op.op);
	}
};

template <>
struct fmt::formatter<klapki::mops> {
	constexpr auto parse(format_parse_context & ctx) { return ctx.begin(); }

	template <typename FormatContext>
	auto format(const klapki::mops & ops, FormatContext & ctx) {
		auto out = ctx.out();

		bool first = true;
		for(auto && el : ops.ops) {
			if(!first)
				out = format_to(out, ", ");
			else
				first = false;

			out = format_to(out, "{}", klapki::op{el});
		}

		return out;
	}
};

template <>
struct fmt::formatter<klapki::detail::initrds_fmt> {
	constexpr auto parse(format_parse_context & ctx) { return ctx.begin(); }

	template <typename FormatContext>
	auto format(const klapki::detail::initrds_fmt & ints, FormatContext & ctx) {
		auto out = ctx.out();

		for(auto && el : ints.initrds)
			out = format_to(out, "{} ", el);
		out = format_to(out, "\"\"");

		return out;
	}
};
