// The MIT License (MIT)

// Copyright (c) 2020 наб <nabijaczleweli@nabijaczleweli.xyz>

// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


#include "state.hpp"
#include <algorithm>
#include <arpa/inet.h>
#include <fmt/format.h>


#define SUB "\x1A"


void klapki::state::stated_config::parse(klapki::state::stated_config & into, const void * _data, std::size_t size) {
	if(size < 2) {
		fmt::print(stderr, "Parsing state: cut off before boot position\n");
		return;
	}

	const char * data = reinterpret_cast<const char *>(_data);
	memcpy(&into.boot_position, data, sizeof(into.boot_position));
	into.boot_position = ntohs(into.boot_position);
	data += sizeof(into.boot_position);
	size -= sizeof(into.boot_position);

	for(;;) {
		const auto variant_end = std::find_if(data, data + size, [](auto b) { return b == '\0'; });
		if(variant_end == data + size) {
			fmt::print(stderr, "Parsing state: cut off after {} variant{}\n", into.variants.size(), into.variants.size() == 1 ? "" : "s");
			data = data + size;
			size = 0;
			break;
		}

		std::string variant{data, static_cast<std::size_t>(variant_end - data)};
		data += variant.size() + 1;  // NUL
		size -= variant.size() + 1;  // NUL

		if(std::find(std::begin(into.variants), std::end(into.variants), variant) != std::end(into.variants)) {
			fmt::print(stderr, "Duplicate variant {}?", variant);
			continue;
		}

		if(variant.empty())
			break;
		else
			into.variants.emplace_back(std::move(variant));
	}

	while(size != 0) {
		stated_config_entry new_entry{};
		if(size <= sizeof(new_entry.bootnum_hint) + sizeof(new_entry.load_option_sha)) {
			fmt::print(stderr, "Parsing state: cut off after {} entr{}\n", into.wanted_entries.size(), into.wanted_entries.size() == 1 ? "y" : "ies");
			break;
		}

		memcpy(&new_entry.bootnum_hint, data, sizeof(new_entry.bootnum_hint));
		new_entry.bootnum_hint = ntohs(new_entry.bootnum_hint);
		data += sizeof(new_entry.bootnum_hint);
		size -= sizeof(new_entry.bootnum_hint);

		memcpy(&new_entry.load_option_sha, data, sizeof(new_entry.load_option_sha));
		data += sizeof(new_entry.load_option_sha);
		size -= sizeof(new_entry.load_option_sha);

		const auto kver_end = std::find_if(data, data + size, [](auto b) { return b == '\0'; });
		if(kver_end == data + size) {
			fmt::print(stderr, "Parsing state: cut off reading kernel version for entry {:04X}\n", new_entry.bootnum_hint);
			break;
		}
		new_entry.version.assign(data, kver_end - data);
		data += new_entry.version.size() + 1;  // NUL
		size -= new_entry.version.size() + 1;  // NUL

		const auto kvar_end = std::find_if(data, data + size, [](auto b) { return b == '\0'; });
		if(kvar_end == data + size) {
			fmt::print(stderr, "Parsing state: cut off reading kernel variant for entry {:04X}\n", new_entry.bootnum_hint);
			break;
		}
		new_entry.variant.assign(data, kvar_end - data);
		data += new_entry.variant.size() + 1;  // NUL
		size -= new_entry.variant.size() + 1;  // NUL

		const auto kdir_end = std::find_if(data, data + size, [](auto b) { return b == '\0'; });
		if(kdir_end == data + size) {
			fmt::print(stderr, "Parsing state: cut off reading kernel image directory for entry {:04X}\n", new_entry.bootnum_hint);
			break;
		}
		new_entry.kernel_dirname.assign(data, kdir_end - data);
		data += new_entry.kernel_dirname.size() + 1;  // NUL
		size -= new_entry.kernel_dirname.size() + 1;  // NUL

		if(size <= sizeof(new_entry.kernel_image_sha)) {
			fmt::print(stderr, "Parsing state: cut off reading kernel image SHA for entry {:04X}\n", new_entry.bootnum_hint);
			break;
		}
		memcpy(&new_entry.kernel_image_sha, data, sizeof(new_entry.kernel_image_sha));
		data += sizeof(new_entry.kernel_image_sha);
		size -= sizeof(new_entry.kernel_image_sha);

		for(;;) {
			const auto idir_end = std::find_if(data, data + size, [](auto b) { return b == '\0'; });
			if(idir_end == data + size) {
				fmt::print(stderr, "Parsing state: cut off after {} initrd{} for entry {:04X}\n", new_entry.bootnum_hint, new_entry.initrd_dirnames.size(),
				           new_entry.initrd_dirnames.size() == 1 ? "" : "s");
				goto end;  // break outer loop; 2020 and C++ does not have this
			}

			std::string idir{data, static_cast<std::size_t>(idir_end - data)};
			data += idir.size() + 1;  // NUL
			size -= idir.size() + 1;  // NUL

			if(idir.empty())
				break;

			shaa_t idir_sha;
			if(size <= idir_sha.size()) {
				fmt::print(stderr, "Parsing state: cut off reading SHA for initrd {} for entry {:04X}\n", new_entry.bootnum_hint, new_entry.initrd_dirnames.size());
				break;
			}
			memcpy(&idir_sha[0], data, idir_sha.size());
			data += idir_sha.size();
			size -= idir_sha.size();

			if(idir == SUB)
				new_entry.initrd_dirnames.emplace_back(nonbase_dirname_t{}, idir_sha);
			else
				new_entry.initrd_dirnames.emplace_back(std::move(idir), idir_sha);
		}

		into.wanted_entries.emplace_back(std::move(new_entry));
	}
end:

	return;
}


std::vector<std::uint8_t> klapki::state::stated_config::serialise() const {
	std::vector<std::uint8_t> ret;

	auto bp  = htons(this->boot_position);
	auto cur = std::copy(reinterpret_cast<std::uint8_t *>(&bp), reinterpret_cast<std::uint8_t *>(&bp) + sizeof(bp), std::back_inserter(ret));

	for(auto && var : this->variants)
		cur = std::copy(var.data(), var.data() + var.size() + 1, cur);
	*cur++ = '\0';

	for(auto && went : this->wanted_entries) {
		auto bh = htons(went.bootnum_hint);

		cur    = std::copy(reinterpret_cast<std::uint8_t *>(&bh), reinterpret_cast<std::uint8_t *>(&bh) + sizeof(bh), cur);
		cur    = std::copy(went.load_option_sha, went.load_option_sha + sizeof(sha_t), cur);
		cur    = std::copy(std::begin(went.version), std::end(went.version), cur);
		*cur++ = '\0';
		cur    = std::copy(std::begin(went.variant), std::end(went.variant), cur);
		*cur++ = '\0';
		cur    = std::copy(std::begin(went.kernel_dirname), std::end(went.kernel_dirname), cur);
		*cur++ = '\0';
		cur    = std::copy(std::begin(went.kernel_image_sha), std::end(went.kernel_image_sha), cur);

		for(auto && [idir, isha] : went.initrd_dirnames) {
			if(idir)
				cur = std::copy(std::begin(*idir), std::end(*idir), cur);
			else
				cur = std::copy(SUB, (SUB) + std::strlen(SUB), cur);
			*cur++ = '\0';

			cur = std::copy(std::begin(isha), std::end(isha), cur);
		}
		*cur++ = '\0';
	}

	return ret;
}
