// The MIT License (MIT)

// Copyright (c) 2020 наб <nabijaczleweli@nabijaczleweli.xyz>

// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


#include "state.hpp"
#include "efi.hpp"
#include <cstdlib>
#include <fmt/format.h>
#include <memory>
#include <optional>
extern "C" {
#include <efivar/efivar.h>
// Doesn't include efivar.h:
#include <efivar/efivar-guids.h>
}


namespace {
	template <class F>
	int iterate_efi_vars(F && func) {
		int status;
		efi_guid_t * iter_guid = nullptr;
		char * iter_name       = nullptr;

		while((status = efi_get_next_variable_name(&iter_guid, &iter_name)))
			func(*iter_guid, iter_name);

		if(status < 0)
			return errno;
		return 0;
	}

	bool operator==(const efi_guid_t & lhs, const efi_guid_t & rhs) noexcept { return !memcmp(&lhs, &rhs, sizeof(efi_guid_t)); }

	bool is_boot_entry(const efi_guid_t & guid, const char * name) noexcept {
		return guid == efi_guid_global && !std::strncmp(name, "Boot", 4) && std::strlen(name) == 8 &&  //
		       std::isxdigit(name[4]) && std::isxdigit(name[5]) && std::isxdigit(name[6]) && std::isxdigit(name[7]);
	}

	bool is_boot_order(const efi_guid_t & guid, const char * name) noexcept { return guid == efi_guid_global && !std::strcmp(name, "BootOrder"); }

	bool is_our_config(const efi_guid_t & guid, const char * name, std::string_view us) noexcept { return guid == klapki::efi_guid_klapki && name == us; }

	template <class F>
	int get_efi_data(const efi_guid_t & guid, const char * name, F && func) {
		std::uint8_t * raw_data{};
		std::size_t size{};
		std::uint32_t attr{};
		if(int res = efi_get_variable(guid, name, &raw_data, &size, &attr))
			return res;

		std::shared_ptr<std::uint8_t[]> data(raw_data, std::free);
		return func(std::move(data), size, attr);
	}

	int get_boot_order(klapki::state::boot_order_flat & bord) {
		return get_efi_data(efi_guid_global, "BootOrder", [&](auto && data, auto size, auto) {
			// endianness?
			bord.order     = std::reinterpret_pointer_cast<std::uint16_t[]>(data);
			bord.order_cnt = size / 2;
			return 0;
		});
	}

	int get_boot_entry(std::map<std::uint16_t, klapki::state::boot_entry> & bents, std::uint16_t num) {
		char name[4 + 4 + 1]{};
		fmt::format_to(name, "Boot{:04X}", num);
		return get_efi_data(efi_guid_global, name, [&](auto && data, auto size, auto attr) {
			bents.emplace(num, klapki::state::boot_entry{std::move(data), size, {}, attr});
			return 0;
		});
	}

	int get_our_config(klapki::state::stated_config & statecfg, std::string_view us) {
		return get_efi_data(klapki::efi_guid_klapki, us.data(), [&](auto && data, auto size, auto) {
			klapki::state::stated_config::parse(statecfg, data.get(), size);
			return 0;
		});
	}
}


std::variant<klapki::state::state, std::string> klapki::state::state::load(std::string_view us) {
	if(!efi_variables_supported())
		return fmt::format("EFI not supported?");


	std::vector<std::uint16_t> boot_entries;
	bool have_boot_order = false;
	bool have_our_config = false;

	if(int res = iterate_efi_vars([&](auto && guid, auto name) {
		   if(is_boot_entry(guid, name))
			   boot_entries.emplace_back(std::strtoul(name + std::strlen("Boot"), nullptr, 16));
		   else if(is_boot_order(guid, name))
			   have_boot_order = true;
		   else if(is_our_config(guid, name, us))
			   have_our_config = true;
	   }))
		return fmt::format("EFI load: iteration: {}", std::strerror(res));  // No threads here, and we avoid a strerror_r() clusterfuck


	boot_order_flat bord{};
	if(have_boot_order) {
		if(get_boot_order(bord) < 0)
			return fmt::format("EFI load: getting BootOrder: {}", std::strerror(errno));
	} else
		fmt::print(stderr, "EFI: no BootOrder?\n");


	std::map<std::uint16_t, boot_entry> entries;
	for(auto bent : boot_entries)
		if(get_boot_entry(entries, bent) < 0)
			return fmt::format("EFI load: getting Boot{:04X}: {}", bent, std::strerror(errno));


	stated_config statecfg{};
	if(have_our_config) {
		if(get_our_config(statecfg, us) < 0)
			return fmt::format("EFI load: getting {}-{}: {}", efi_guid_klapki_s, us, std::strerror(errno));
	} else
		fmt::print(stderr, "EFI load: no config for this host ({}) found; going to the top\n", us);

	return state{std::move(bord), std::move(entries), std::move(statecfg)};
}


bool klapki::state::stated_config_entry::operator==(const klapki::state::stated_config_entry & other) const noexcept {
	return this->bootnum_hint == other.bootnum_hint &&                                     //
	       !std::memcmp(this->load_option_sha, other.load_option_sha, sizeof(sha_t)) &&    //
	       this->version == other.version &&                                               //
	       this->variant == other.variant &&                                               //
	       this->kernel_dirname == other.kernel_dirname &&                                 //
	       !std::memcmp(this->kernel_image_sha, other.kernel_image_sha, sizeof(sha_t)) &&  //
	       this->initrd_dirnames == other.initrd_dirnames;
}

bool klapki::state::stated_config::operator==(const klapki::state::stated_config & other) const noexcept {
	return this->boot_position == other.boot_position && this->variants == other.variants && this->wanted_entries == other.wanted_entries;
}
