// SPDX-License-Identifier: MIT
/*! klapki::config::read(): step 1
 * Parse the environment and arguments:
 *   $KLAPKI_HOST OR ${KLAPKI_WISDOM-/etc/klapki}/klapki (first line) OR /etc/machine-id (first line) OR hostname -> config::host
 *   $KLAPKI_WISDOM normalised to have a slash at the end if nonempty or "/etc/klapki/"                           -> config::wisdom_root_raw
 *   -v                                                                                                           -> config::verbose
 *   !-n                                                                                                          -> config::commit
 *   in order p in [/boot/efi, /efi, /boot]: if p is a mount-point: "p"; if none: error                           -> config::esp
 *   (${KLAPKI_WISDOM-/etc/klapki}/efi-root (first line) -> normalised to "\a\b\c") OR "\klapki"                  -> config::efi_root_raw
 *   argv -> klapki::op::from_cmdline()                                                                           -> config::ops
 *
 * And for each -E do efi_set_verbose(efi_get_verbose() + 1, nullptr).
 */


#include "config.hpp"
#include "util.hpp"
#include <algorithm>
#include <cstdlib>
#include <fmt/format.h>
#include <fstream>
#include <getopt.h>
#include <limits.h>
#include <sys/stat.h>
#include <sys/utsname.h>
#include <unistd.h>
extern "C" {
#pragma GCC diagnostic push
#if __clang__
#pragma GCC diagnostic ignored "-Wgnu-zero-variadic-macro-arguments"
#endif
#pragma GCC diagnostic ignored "-Wvariadic-macros"
#include <efivar.h>
#pragma GCC diagnostic pop
}

using namespace std::literals;


#define TRY(...)                                       \
	({                                                   \
		auto ret = __VA_ARGS__;                            \
		if(auto err = std::get_if<std::string>(&ret); err) \
			return std::move(*err);                          \
		std::move(std::get<0>(ret));                       \
	})


namespace {
	std::optional<std::string> readline(const char * from) {
		std::ifstream in(from);
		if(!in)
			return std::nullopt;

		std::string ret;
		std::getline(in, ret);
		return ret;
	}

	std::string get_host(const std::string_view & wisdom_root) {
		if(auto host = std::getenv("KLAPKI_HOST"))
			return host;
		else {
			if(auto host = readline((std::string{wisdom_root} += "host").c_str()))
				return std::move(*host);

			if(auto mid = readline("/etc/machine-id"))
				return std::move(*mid);

			struct utsname utsname;
			uname(&utsname);
			return utsname.nodename;
		}
	}

	std::variant<std::string_view, std::string> get_wisdom_root() {
		if(auto host = std::getenv("KLAPKI_WISDOM")) {
			std::string_view ret{host};
			if(!ret.empty() && ret.back() != '/')
				return std::string{ret} += '/';
			else
				return ret;
		} else
			return "/etc/klapki/"sv;
	}

	std::variant<std::string_view, std::string> find_esp(const char * argv0) {
		struct stat boot, boot_efi;
		if(stat("/boot", &boot) == -1)
			return fmt::format("{}: stat(\"{}\"): {}", argv0, "/boot"sv, std::strerror(errno));
		if(stat("/boot/efi", &boot_efi) == -1) {
			if(errno != ENOENT)
				return fmt::format("{}: stat(\"{}\"): {}", argv0, "/boot/efi"sv, std::strerror(errno));
			goto try_efi;
		}

		if(boot.st_dev != boot_efi.st_dev)
			return "/boot/efi"sv;

	try_efi:
		struct stat root, efi;
		if(stat("/", &root) == -1)
			return fmt::format("{}: stat(\"{}\"): {}", argv0, "/"sv, std::strerror(errno));
		if(stat("/efi", &efi) == -1) {
			if(errno != ENOENT)
				return fmt::format("{}: stat(\"{}\"): {}", argv0, "/efi"sv, std::strerror(errno));
			goto try_boot;
		}

		if(root.st_dev != efi.st_dev)
			return "/efi"sv;

	try_boot:
		if(root.st_dev != boot.st_dev)
			return "/boot"sv;

		return fmt::format(fgettext("{}: ESP discovery: none of /boot/efi, /efi, /boot are mount-points?"), argv0);
	}


	std::variant<std::string_view, std::string> get_efi_root(const std::string_view & wisdom_root) {
		auto configured = readline((std::string{wisdom_root} += "efi-root").c_str());
		if(!configured)
			if(auto root = std::getenv("KLAPKI_EFI_ROOT"))
				configured = root;
		if(!configured)
			return "\\klapki"sv;

		auto && root = *configured;
		std::replace(std::begin(root), std::end(root), '/', '\\');
		root.insert(0, 1, '\\');
		root.erase(std::unique(std::begin(root), std::end(root), [](auto l, auto r) { return l == '\\' && r == '\\'; }), std::end(root));
		if(!root.empty() && root.back() == '\\')
			root.pop_back();
		if(root.size() == 1)  // == "\\"sv
			root = {};
		return std::move(root);
	}
}


std::variant<klapki::config, std::string> klapki::config::read(int argc, char * const * argv) {
	const char * argv0 = argv[0];

	bool verbose  = false;
	bool verybose = false;
	bool commit   = true;
#if KLAPKI_NO_ZLIB
	bool compress = false;
#else
	bool compress = true;
#endif
	for(int arg; (arg = getopt(argc, argv, "+nvVECh")) != -1;)
		switch(arg) {
			case 'v':
				verbose = true;
				break;

			case 'V':
				verybose = true;
				break;

			case 'E':
				efi_set_verbose(efi_get_verbose() + 1, nullptr);
				break;

			case 'n':
				commit = false;
				break;

			case 'C':
				compress = false;
				break;

			case 'h':
			default:
				// Remember to sync to klapki(8)
				return fmt::format(fgettext("klapki {}\n"
				                            "Usage: {} [-nvVECh]… [op [arg…]]…\n"
				                            "\n"
				                            "Flags:\n"
				                            "  -n\tDon't commit\n"
				                            "  -v\tVerbose operation\n"
				                            "  -V\tAdd {{dump}}s around all ops\n"
				                            "  -E\tIncrease libefivar verbosity level\n"
				                            "  -C\tDon't (de)compress state\n"
				                            "  -h\tShow this help\n"
				                            "\n"
				                            "Environment:\n"
				                            "  KLAPKI_WISDOM=  \tReplaces /etc/klapki\n"
				                            "  KLAPKI_HOST=    \tReplaces contents of /etc/klapki/host or /etc/machine-id or the hostname\n"
				                            "  KLAPKI_EFI_ROOT=\tReplaces contents of /etc/klapki/efi-root or \"\\klapki\"\n"
				                            "\n"
				                            "Recognised ops:\n"
				                            "\tdump\n"
				                            "\tbootpos <position>\n"
				                            "\taddkernel <version> <image> [initrd…] <\"\">\n"
				                            "\tdelkernel <version>\n"
				                            "\taddvariant <variant>\n"
				                            "\tdelvariant <variant>"),
				                   KLAPKI_VERSION, argv0);
		}

	config ret{{}, verbose, commit, compress, TRY(find_esp(argv0)), {}, {}, get_wisdom_root()};

	if(verybose)
		ret.ops.emplace_back(ops::dump{});
	for(argv += optind; *argv;) {
		auto pop = op::from_cmdline(argv0, argv);
		if(auto err = std::get_if<std::string>(&pop))
			return std::move(*err);
		else {
			ret.ops.emplace_back(std::move(std::get<ops::op_t>(pop)));
			if(verybose)
				ret.ops.emplace_back(ops::dump{});
		}
	}

	ret.host         = get_host(ret.wisdom_root());
	ret.efi_root_raw = get_efi_root(ret.wisdom_root());
	return ret;
}
