// SPDX-License-Identifier: MIT
/*! klapki::context::context::commit(): step 10
 * Copy and delete files to and in the ESP:
 *   * create leading directories for context::context::our_kernels::image_path (::first) in the ESP
 *   * also do this for context::context::deleted_files (::first) but save the created directories for removal later
 *   * open and read the directories corresponding to state::state::statecfg::wanted_entries' ::kernel_dirname and ::initrd_dirnames,
 *     keeping the filenames' case but matching insensitively since we mangle the case for the image
 *   * for the join of context::context::our_kernels with output state::state::statecfg::wanted_entries:
 *     1. do_copy()
 *        (look up the "real" spelling of the basename from the cache,
 *         if a file with a matching name was already copied to the directory, copy the SHA, warn, and exit
 *         otherwise copy_file() (open the source file
 *                                SHA1 it, update the SHA
 *                                if it's the same as the known previous SHA and destination exists: return
 *                                create and copy source to the destination)
 *         save the filename as already copied)
 *        with source            = state::state::statecfg::wanted_entries::kernel_dirname
 *             dest              = context::context::our_kernels::image_path::first
 *             basename          =                                          ::second
 *             known/updated SHA = state::state::statecfg::wanted_entries::kernel_image_sha
 *     2. do_copy() for each initrd
 *        with source            = state::state::statecfg::wanted_entries::kernel_dirname or ::initrd_dirnames::first if non-nullopt
 *             dest              = context::context::our_kernels::image_path::first       or ::initrd_paths::first    if non-nullopt
 *             basename          =                              ::initrd_paths::second
 *             known/updated SHA = state::state::statecfg::initrd_dirnames::second
 *   * delete context::context::deleted_files from the ESP /if/ they weren't copied to
 *   * delete the stashed-after-creation directories for ::deleted_files from the ESP (ignore errors)
 *
 * The context::context is read-only, the output state::state::statecfg is only written to to update the file SHAs.
 */


#include "config.hpp"
#include "context.hpp"
#include "context_detail.hpp"
#include "vore-dir"
#include "vore-file"
#include "vore-mmap"
#include <algorithm>
#include <dirent.h>
#include <fcntl.h>
#include <set>
#include <strings.h>
#include <sys/mman.h>
#include <sys/sendfile.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>


#define TRY(...)                                       \
	({                                                   \
		auto ret = __VA_ARGS__;                            \
		if(auto err = std::get_if<std::string>(&ret); err) \
			return std::move(*err);                          \
		std::move(std::get<0>(ret));                       \
	})

#define TRY_OPT(...)              \
	if(auto err = __VA_ARGS__; err) \
		return err;


namespace {
	struct caseless {
		using is_transparent = void;

		bool operator()(const std::string & l, const std::string & r) const noexcept { return strcasecmp(l.c_str(), r.c_str()) < 0; }
		bool operator()(const std::string & l, const char * r) const noexcept { return strcasecmp(l.c_str(), r) < 0; }
		bool operator()(const std::string_view & l, const std::string_view & r) const noexcept { return strcasecmp(l.data(), r.data()) < 0; }
		bool operator()(const std::string_view & l, const char * r) const noexcept { return strcasecmp(l.data(), r) < 0; }
	};

	/// https://stackoverflow.com/a/2180157/2851815 says FreeBSD and Darwin have fcopyfile(3),
	/// but the only reference I could find was (a) copies of that answer and (b) Apple documentation, which wasn't helpful.
	///
	/// Plus, I've no idea if libefivar even works on Berkeley distros; it's got separate linux*.c implementations, but.
	///
	/// Just use sendfile(2) here and potentially ifdef for non-Linux later.
	std::variant<bool, std::string> copy_file(int srcdir, int destdir, const char * basename, klapki::sha_t cursha, bool verbose) {
		vore::file::fd srcfd{basename, O_RDONLY | O_CLOEXEC, 0, srcdir};
		if(srcfd == -1)
			return fmt::format(fgettext("Couldn't open {} for reading: {}\n"), basename, std::strerror(errno));

		struct stat src_sb;
		fstat(srcfd, &src_sb);

		klapki::sha_t insha{};
		{
			vore::file::mapping map{nullptr, static_cast<std::size_t>(src_sb.st_size), PROT_READ, MAP_PRIVATE, srcfd};
			if(!map)
				return fmt::format(fgettext("Couldn't mmap() source {}: {}\n"), basename, std::strerror(errno));

			klapki::SHA1(map->data(), map->size(), insha);
		}
		if(!std::memcmp(cursha, insha, sizeof(klapki::sha_t))) {
			if(verbose)
				// filename, SHA
				fmt::print(fgettext("{} unchanged ({})\n"), basename, klapki::context::detail::sha_f{cursha});

			if(faccessat(destdir, basename, F_OK, 0) == -1) {
				if(errno != ENOENT)
					return fmt::format(fgettext("Couldn't stat() destination {}: {}\n"), basename, std::strerror(errno));
				// Still copy if missing
			} else
				return false;
		}

		if(verbose)
			// filename, SHAs
			fmt::print(fgettext("{} changed ({} -> {})\n"), basename, klapki::context::detail::sha_f{cursha}, klapki::context::detail::sha_f{insha});
		std::memcpy(cursha, insha, sizeof(klapki::sha_t));

		vore::file::fd destfd{basename, O_WRONLY | O_CREAT | O_TRUNC | O_CLOEXEC, 0666, destdir};
		if(destfd == -1)
			return fmt::format(fgettext("Couldn't open {} for writing: {}\n"), basename, std::strerror(errno));

		off_t offset = 0;
		for(ssize_t rd; (rd = sendfile(destfd, srcfd, &offset, src_sb.st_size));)
			if(rd == -1)
				return fmt::format(fgettext("Couldn't copy to {} (offset {}): {}\n"), basename, offset, std::strerror(errno));

		return true;
	}
}


std::optional<std::string> klapki::context::context::commit(const config & cfg, state::state & state) const {
	vore::file::fd esp_fd{cfg.esp.data(), O_PATH | O_DIRECTORY | O_CLOEXEC};
	if(esp_fd == -1)
		return fmt::format(fgettext("Couldn't open ESP ({}): {}\n"), cfg.esp, std::strerror(errno));

	std::map<std::string_view, vore::file::fd, caseless> esp_dirs;
	std::set<std::string> deleted_dir_names;
	auto adddir = [&](auto && ddir, auto && cbk) -> std::optional<std::string> {
		if(!esp_dirs.contains(ddir)) {
			auto dir = ddir;
			std::replace(std::begin(dir), std::end(dir), '\\', '/');
			dir.erase(std::unique(std::begin(dir), std::end(dir), [](auto l, auto r) { return l == '/' && r == '/'; }), std::end(dir));
			if(dir.back() == '/')
				dir.pop_back();
			dir.erase(0, dir.find_first_not_of('/'));

			for(auto slash_idx = dir.find('/'); slash_idx != std::string::npos; slash_idx = dir.find('/', slash_idx + 1)) {
				dir[slash_idx] = '\0';
				if(mkdirat(esp_fd, dir.c_str(), 0777) == -1 && errno != EEXIST)
					return fmt::format(fgettext("Couldn't create {} under ESP ({}): {}\n"), dir.c_str(), cfg.esp, std::strerror(errno));
				dir[slash_idx] = '/';
			}
			if(mkdirat(esp_fd, dir.c_str(), 0777) == -1 && errno != EEXIST)
				return fmt::format(fgettext("Couldn't create {} under ESP ({}): {}\n"), dir, cfg.esp, std::strerror(errno));

			vore::file::fd fd{dir.c_str(), O_PATH | O_DIRECTORY | O_CLOEXEC, 0, esp_fd};
			if(fd == -1)
				return fmt::format(fgettext("Couldn't open {} under ESP ({}): {}\n"), dir, cfg.esp, std::strerror(errno));
			esp_dirs.emplace(ddir, std::move(fd));
			cbk(std::move(dir));
		}

		return {};
	};
	for(auto && kern : this->our_kernels)
		TRY_OPT(adddir(kern.second.image_path.first, [](auto &&) {}));
	for(auto && [ddir, _] : this->deleted_files)
		TRY_OPT(adddir(ddir, [&](auto && dname) { deleted_dir_names.emplace(std::move(dname)); }));


	std::map<detail::bad_cow, std::pair<vore::file::DIR, std::set<std::string, caseless>>> source_dirs;

	for(auto && skern : state.statecfg.wanted_entries) {
		source_dirs.emplace(detail::bad_cow{skern.kernel_dirname}, decltype(source_dirs)::mapped_type{});
		for(auto && [initrd_dirname, _] : skern.initrd_dirnames)
			if(initrd_dirname)
				source_dirs.emplace(detail::bad_cow{*initrd_dirname}, decltype(source_dirs)::mapped_type{});
	}

	for(auto && [path, dir_ents] : source_dirs) {
		dir_ents.first = {path.get().data()};
		if(!dir_ents.first)
			return fmt::format(fgettext("Couldn't open {}: {}\n"), path.get(), std::strerror(errno));

		while(const auto ent = readdir(dir_ents.first))
			dir_ents.second.emplace(ent->d_name);
	}


	std::map<std::pair<int, std::string>, const std::uint8_t *> already_copied;
	auto do_copy = [&](auto bootnum, auto && srcdir, auto && destdir, const auto * basename, auto cursha) -> std::optional<std::string> {
		auto destfd = esp_dirs.find(destdir);
		if(destfd == std::end(esp_dirs))
			return fmt::format("ESP dir {} not in cache?", destfd->first);

		auto srcfd = source_dirs.find(detail::bad_cow{srcdir});
		if(srcfd == std::end(source_dirs))
			return fmt::format("Source dir {} not in cache?", srcdir);

		if(const auto entry = srcfd->second.second.find(*basename); entry != srcfd->second.second.end())
			basename = &*entry;

		if(auto ac = already_copied.find(std::pair{static_cast<int>(destfd->second), *basename}); ac != std::end(already_copied)) {
			if(cfg.verbose)
				// Entry {1234}: already copied {vmlinuz-ver} from {/boot} to {\klapki\123\ver}
				fmt::print(fgettext("Entry {:04X}: already copied {} from {} to {}\n"), bootnum, *basename, srcdir, destfd->first);
			std::memcpy(cursha, ac->second, sizeof(klapki::sha_t));
			return {};
		}

		if(cfg.verbose)
			// Entry {1234}: copying {vmlinuz-ver} from {/boot} to {\klapki\123\ver}
			fmt::print(fgettext("Entry {:04X}: copying {} from {} to {}\n"), bootnum, *basename, srcdir, destfd->first);
		if(TRY(copy_file(dirfd(srcfd->second.first), destfd->second, basename->c_str(), cursha, cfg.verbose)))
			// Entry {1234}: copied {vmlinuz-ver} from {/boot} to {\klapki\123\ver}
			fmt::print(fgettext("Entry {:04X}: copied {} from {} to {}\n"), bootnum, *basename, srcdir, destfd->first);
		already_copied.emplace(std::pair{static_cast<int>(destfd->second), *basename}, cursha);
		return {};
	};

	for(auto && [bootnum, kern] : this->our_kernels) {
		auto skern = std::find_if(std::begin(state.statecfg.wanted_entries), std::end(state.statecfg.wanted_entries),
		                          [&](auto && skern) { return skern.bootnum_hint == bootnum; });
		if(skern == std::end(state.statecfg.wanted_entries))
			throw __func__;

		TRY_OPT(do_copy(bootnum, skern->kernel_dirname, kern.image_path.first, &kern.image_path.second, skern->kernel_image_sha));

		if(skern->initrd_dirnames.size() != kern.initrd_paths.size())
			throw __func__;

		std::string_view sprev = skern->kernel_dirname;
		std::string_view dprev = kern.image_path.first;
		for(std::size_t i = 0; i < skern->initrd_dirnames.size(); ++i) {
			auto && [sidir, ssha]   = skern->initrd_dirnames[i];
			auto && [didir, dibase] = kern.initrd_paths[i];

			if(sidir)
				sprev = *sidir;
			if(didir)
				dprev = *didir;
			TRY_OPT(do_copy(bootnum, sprev, dprev, &dibase, &ssha[0]));
		}
	}


	for(auto && [ddir, dfile] : this->deleted_files) {
		auto destfd = esp_dirs.find(ddir);
		if(destfd == std::end(esp_dirs))
			return fmt::format("ESP dir {} not in cache?", ddir);

		if(already_copied.find(std::pair{static_cast<int>(destfd->second), dfile}) != std::end(already_copied)) {
			if(cfg.verbose)
				// Not removing {vmlinuz-ver} from {\klapki\123\ver} after ...
				fmt::print(fgettext("Not removing {} from {} after having copied it there\n"), dfile, ddir);
			continue;
		}

		// Removing {vmlinuz-ver} from {\klapki\123\ver}
		fmt::print(fgettext("Removing {} from {}\n"), dfile, ddir);
		if(unlinkat(destfd->second, dfile.c_str(), 0) == -1 && unlinkat(destfd->second, dfile.c_str(), AT_REMOVEDIR) == -1)
			// Removing {vmlinuz-ver} from {\klapki\123\ver}: {errno}
			return fmt::format(fgettext("Removing {} from {}: {}"), dfile, ddir, std::strerror(errno));
	}

	for(auto && ddir : deleted_dir_names) {
		if(cfg.verbose)
			// Cleaning up {vmlinuz-ver} from {\klapki\123\ver}
			fmt::print(fgettext("Cleaning up {} from {}\n"), ddir, cfg.esp);
		if(unlinkat(esp_fd, ddir.c_str(), AT_REMOVEDIR) == -1)
			if(cfg.verbose)
				// Cleaning up {vmlinuz-ver} from {\klapki\123\ver}: {errno}
				fmt::print(fgettext("Removing {} from {}: {}\n"), ddir, cfg.esp, std::strerror(errno));
	}


	return {};
}
