// SPDX-License-Identifier: MIT
/*! klapki::context::resolve_state_context(): step 3
 * Parse input_state klapki::state::state into a structured version:
 *   input state::entries -> Boot1234 SHA1s computed
 *   input state::statecfg::wanted_entries -> copy, matching state::statecfg::wanted_entries::load_option_sha to state::entries::load_option_sha:
 *     1. by state::statecfg::wanted_entries::bootnum_hint if SHAs match
 *     2. then globally by SHA (and updating the resulting state::statecfg::wanted_entries::bootnum_hint)
 *     3. dropping it from wanted_entries if doesn't exist
 *   input state::boot_order boot_order_flat -> boot_order_structured:
 *     entries that also figure in the result state::statecfg::wanted_entries::bootnum_hint go to ::ours,
 *     others go to ::foreign
 */


#include "config.hpp"
#include "context.hpp"
#include "context_detail.hpp"
#include "util.hpp"
#include <algorithm>
#include <cstring>
#include <fmt/format.h>
#include <iterator>
#include <numeric>


std::variant<klapki::state::state, std::string> klapki::context::resolve_state_context(const config & cfg, const state::state & input_state) {
	auto entries = input_state.entries;
	for(auto & [_, bent] : entries)
		klapki::SHA1(bent.load_option.get(), bent.load_option_len, bent.load_option_sha);


	state::stated_config statecfg{input_state.statecfg.boot_position, input_state.statecfg.variants, {}};
	statecfg.wanted_entries.reserve(input_state.statecfg.wanted_entries.size());


	// Match wanted entries to boot entries
	std::vector<std::pair<sha_t, std::uint16_t>> remaps;
	std::copy_if(std::begin(input_state.statecfg.wanted_entries), std::end(input_state.statecfg.wanted_entries), std::back_inserter(statecfg.wanted_entries),
	             [&](auto && went) {
		             if(auto bent = entries.find(went.bootnum_hint); bent != std::end(entries)) {
			             if(!std::memcmp(bent->second.load_option_sha, went.load_option_sha, sizeof(sha_t))) {
				             if(cfg.verbose)
					             fmt::print(fgettext("Entry {:04X} matches\n"), went.bootnum_hint);
				             return true;
			             } else
				             fmt::print(stderr, fgettext("Entry {:04X}: mismatched hash, searching elsewhere\n"), went.bootnum_hint);
		             } else
			             fmt::print(stderr, fgettext("Entry {:04X} doesn't exist; moved?, searching elsewhere\n"), went.bootnum_hint);

		             if(auto bent = std::find_if(begin(entries), end(entries),
		                                         [&](const auto & bent) { return !std::memcmp(bent.second.load_option_sha, went.load_option_sha, sizeof(sha_t)); });
		                bent != end(entries)) {
			             fmt::print(stderr, fgettext("Found entry formerly {:04X} at {:04X}.\n"), went.bootnum_hint, bent->first);

			             std::pair<sha_t, std::uint16_t> remap;
			             std::memcpy(remap.first, went.load_option_sha, sizeof(sha_t));
			             remap.second = bent->first;
			             remaps.emplace_back(std::move(remap));

			             return true;
		             } else {
			             // SHA
			             fmt::print(stderr, fgettext("Entry formerly {:04X} ({}) not found. Abandoning.\n"), went.bootnum_hint, detail::sha_f{went.load_option_sha});
			             return false;
		             }
	             });

	for(auto && [sha, bootnum] : remaps)
		if(auto went = std::find_if(std::begin(statecfg.wanted_entries), std::end(statecfg.wanted_entries),
		                            [&](auto && went) { return !std::memcmp(went.load_option_sha, sha, sizeof(sha_t)); });
		   went != std::end(statecfg.wanted_entries))
			went->bootnum_hint = bootnum;


	// Structure boot order
	auto boot_order = std::visit(overload{[&](const state::boot_order_flat & bof) {
		                                      state::boot_order_structured ret{};

		                                      for(auto cur = bof.order.get(); cur != bof.order.get() + bof.order_cnt; ++cur)
			                                      if(std::any_of(std::begin(statecfg.wanted_entries), std::end(statecfg.wanted_entries),
			                                                     [&](auto && went) { return went.bootnum_hint == *cur; }))
				                                      ret.ours.emplace_back(*cur);
			                                      else
				                                      ret.foreign.emplace_back(*cur);

		                                      return ret;
	                                      },
	                                      [](const state::boot_order_structured & bos) { return bos; }},
	                             input_state.order);


	return state::state{std::move(boot_order), std::move(entries), std::move(statecfg)};
}
