// SPDX-License-Identifier: MIT


#include "ops.hpp"
#include <cstring>
#include <libintl.h>
#include <limits>

using namespace std::literals;


namespace {
	std::variant<klapki::ops::op_t, std::string> bootpos_op(const char * argv0, char * const *& argv) {
		char * end;
		// strtoul(3): In particular, if *nptr is not '\0' but **endptr is '\0' on return, the entire string is valid.
		if(!argv[0] || argv[0][0] == '\0')
			return fmt::format(fgettext("{}: bootpos requires a position"), argv0);
		errno               = 0;
		const auto position = argv[0];
		++argv;
		const auto num = std::strtoull(position, &end, 0);
		if((num > 0xFFFF && (errno = EOVERFLOW)) || *end != '\0')
			// {klapki}: bootpos position {user's argument}: {error}
			return fmt::format(fgettext("{}: bootpos position {}: {}"), argv0, position, std::strerror(errno ? errno : EINVAL));

		return klapki::ops::bootpos{static_cast<std::uint16_t>(num)};
	}

	template <class T>
	std::variant<klapki::ops::op_t, std::string> oneval_op(const char * argv0, char * const *& argv, const char * name, const char * aname) {
		if(!argv[0])
			// {klapki}: {delkernel|{add|del}variant} requires {version|variant}
			return fmt::format(fgettext("{}: {} requires {}"), argv0, name, aname);
		return T{argv++[0]};
	}

	std::variant<klapki::ops::op_t, std::string> addkernel_op(const char * argv0, char * const *& argv) {
		klapki::ops::addkernel ret;

		if(!argv[0])
			return fmt::format(fgettext("{}: addkernel requires <version> <image> [initrd…] <\"\">"), argv0);
		ret.version = argv[0];
		++argv;

		if(!argv[0])
			// {klapki}: addkernel {version} ... (as part of the arguments)
			return fmt::format(fgettext("{}: addkernel {} requires <image> [initrd…] <\"\">"), argv0, ret.version);
		if(argv[0][0] == '\0')
			// {klapki}: addkernel {version} ... (as part of the arguments)
			return fmt::format(fgettext("{}: addkernel {} image can't be empty"), argv0, ret.version);
		ret.image = argv[0];
		++argv;

		while(argv[0]) {
			const auto initrd = argv[0];
			++argv;
			if(initrd[0] == '\0')
				break;
			ret.initrds.emplace_back(initrd);
		}
		if(argv[-1][0] != '\0')
			// {klapki}: addkernel {version} {image file} ... (as part of the arguments)
			return fmt::format(fgettext("{}: addkernel {} {} initrd list must end with empty argument"), argv0, ret.version, ret.image);

		return ret;
	}
}


std::variant<klapki::ops::op_t, std::string> klapki::op::from_cmdline(const char * argv0, char * const *& argv) {
	const std::string_view opname = *argv++;
	if(opname == "dump"sv)
		return ops::dump{};
	else if(opname == "bootpos"sv)
		return bootpos_op(argv0, argv);
	else if(opname == "addkernel"sv)
		return addkernel_op(argv0, argv);
	else if(opname == "delkernel"sv)
		// only for "... requires {}" above
		return oneval_op<ops::delkernel>(argv0, argv, "delkernel", gettext("version"));
	else if(opname == "addvariant"sv)
		// only for "... requires {}" above
		return oneval_op<ops::addvariant>(argv0, argv, "addvariant", gettext("variant"));
	else if(opname == "delvariant"sv)
		// only for "... requires {}" above
		return oneval_op<ops::delvariant>(argv0, argv, "delvariant", gettext("variant"));
	else
		return fmt::format(fgettext("{}: unknown op {}"), argv0, opname);
}


std::optional<std::string> klapki::op::execute(const klapki::ops::op_t & op, const config & cfg, state::state & state, context::context & context) {
	return std::visit([&](const auto & o) { return ops::execute(o, cfg, state, context); }, op);
}
