// SPDX-License-Identifier: MIT


#pragma once


#include "sha1.hpp"
#include "util.hpp"
#include <array>
#include <cstdint>
#include <fmt/format.h>
#include <libintl.h>
#include <map>
#include <optional>
#include <string>
#include <variant>
#include <vector>

namespace klapki {
	struct config;
}


namespace klapki::state {
	using shaa_t            = std::array<std::uint8_t, sizeof(sha_t)>;
	using nonbase_dirname_t = std::optional<std::string>;  // Use SUB to copy from previous entry

	struct boot_order_flat {
		std::shared_ptr<std::uint16_t[]> order;
		std::size_t order_cnt;
	};
	struct boot_order_structured {
		std::vector<std::uint16_t> foreign, ours;
	};
	using boot_order_t = std::variant<boot_order_flat, boot_order_structured>;

	struct boot_entry {
		std::shared_ptr<std::uint8_t[]> load_option;
		std::size_t load_option_len;
		sha_t load_option_sha;

		std::uint32_t attributes;
	};

	struct stated_config_entry {
		/// Hint initially, then pointer
		std::uint16_t bootnum_hint;  // big-endian
		sha_t load_option_sha;
		std::string version;         // NUL-terminated
		std::string variant;         // NUL-terminated
		std::string kernel_dirname;  // on rootfs; NUL-terminated
		sha_t kernel_image_sha;
		std::vector<std::pair<nonbase_dirname_t, shaa_t>> initrd_dirnames;  // on rootfs; entries NUL-terminated; list empty-terminated w/o SHA


		bool operator==(const stated_config_entry & other) const noexcept;
	};

	struct stated_config {
		std::uint16_t boot_position;        // big-endian
		std::vector<std::string> variants;  // entries NUL-terminated; list empty-terminated; kept unique, order-preserving
		std::vector<stated_config_entry> wanted_entries;

		static void parse(stated_config & into, bool may_decompress, const void * data, std::size_t size);

		std::vector<std::uint8_t> serialise(bool may_compress) const;


		bool operator==(const stated_config & other) const noexcept;
		inline bool operator!=(const stated_config & other) const noexcept { return !(*this == other); }
	};

	struct state {
		boot_order_t order;
		std::map<std::uint16_t, boot_entry> entries;
		stated_config statecfg;

		std::optional<std::string> commit(const char * us, bool may_compress_statecfg, const state & original_state) const;

		static std::variant<state, std::string> load(const std::string_view & us, bool may_decompress_statecfg);
	};
}


template <>
struct fmt::formatter<klapki::state::boot_order_t> {
	template <class OItr>
	static OItr write(OItr out, const std::uint16_t * first, std::size_t cnt) {
		for(; cnt--; ++first) {
			out = format_to(out, "{:04X}", *first);
			if(cnt != 0)
				*out++ = ',';
		}
		return out;
	}

	constexpr auto parse(format_parse_context & ctx) { return ctx.begin(); }

	template <typename FormatContext>
	auto format(const klapki::state::boot_order_t & bord, FormatContext & ctx) {
		auto out = ctx.out();

		std::visit(klapki::overload{
		               [&](const klapki::state::boot_order_flat & bord) { out = write(out, bord.order.get(), bord.order_cnt); },
		               [&](const klapki::state::boot_order_structured & bord) {
			               *out++ = '[';
			               out    = write(out, bord.foreign.data(), bord.foreign.size());
			               *out++ = ']';
			               *out++ = ' ';
			               *out++ = '&';
			               *out++ = ' ';
			               *out++ = '{';
			               out    = write(out, bord.ours.data(), bord.ours.size());
			               *out++ = '}';
		               },
		           },
		           bord);

		return out;
	}
};

template <>
struct fmt::formatter<klapki::state::stated_config_entry> {
	constexpr auto parse(format_parse_context & ctx) { return ctx.begin(); }

	template <typename FormatContext>
	auto format(const klapki::state::stated_config_entry & ent, FormatContext & ctx) {
		auto out = ctx.out();

		out = format_to(out, "{{ {:04X}, \"{}\", {}{}{}, \"{}\", [",  //
		                ent.bootnum_hint,                             //
		                ent.version,                                  // default variant
		                ent.variant.empty() ? "" : "\"", ent.variant.empty() ? gettext("(default)") : std::string_view{ent.variant},
		                ent.variant.empty() ? "" : "\"",  //
		                ent.kernel_dirname);

		bool first = true;
		for(auto && [el, _] : ent.initrd_dirnames) {
			if(!first)
				out = format_to(out, ", ");
			else
				first = false;

			if(el)
				*out++ = '\"';
			out = format_to(out, "{}", el ? std::string_view{*el} : "←");
			if(el)
				*out++ = '\"';
		}

		out = format_to(out, "] }}");

		return out;
	}
};
