// SPDX-License-Identifier: MIT


// HD(1,GPT,48520351-6c2c-4617-a8d1-f353b750ef98,0x800,0x76800)
#define INIT_HD                                                                                                                                     \
	efi_guid_t GUID{0x48520351, 0x6c2c, 0x4617, htobe16(0xa8d1), {0xf3, 0x53, 0xb7, 0x50, 0xef, 0x98}};                                               \
	efidp_make_hd(reinterpret_cast<std::uint8_t *>(&HD), sizeof(HD), 1, 0x800, 0x76800, reinterpret_cast<std::uint8_t *>(&GUID), EFIDP_HD_FORMAT_GPT, \
	              EFIDP_HD_SIGNATURE_GUID);
#define NAME "babtop"
#include "everything.hpp"

// Slightly degraded (the boot entries with the debug variant didn't get saved to the firmware, I think it's just full flash?).
// This means we ought to re-create the state in the state variable precisely (sans the file SHAs).
//
// # efibootmgr -u
// BootCurrent: 0005
// Timeout: 1 seconds
// BootOrder: 0005,0004,0001,0002,0003
// Boot0001* UEFI:CD/DVD Drive	BBS(129,,0x0)
// Boot0002* UEFI:Removable Device	BBS(130,,0x0)
// Boot0003* UEFI:Network Device	BBS(131,,0x0)
// Boot0004* Debian GNU/Linux trixie/sid with Linux 6.1.0-7-amd64
// HD(1,GPT,48520351-6c2c-4617-a8d1-f353b750ef98,0x800,0x76800)/File(\KLAPKI\731B69F0DAC147EFADFED92F12712736\6.1.0-7-AMD64\VMLINUZ-6.1.0-7-AMD64)initrd=\klapki\731b69f0dac147efadfed92f12712736\6.1.0-7-amd64\initrd.img-6.1.0-7-amd64
// root=zfs:AUTO fbcon=rotate:3 intel_iommu=on zfs.zfs_arc_max=12884901888 quiet module.sig_enforce=1 Boot0005* Debian GNU/Linux trixie/sid with
// Linux 6.1.0-9-amd64
// HD(1,GPT,48520351-6c2c-4617-a8d1-f353b750ef98,0x800,0x76800)/File(\KLAPKI\731B69F0DAC147EFADFED92F12712736\6.1.0-9-AMD64\VMLINUZ-6.1.0-9-AMD64)initrd=\klapki\731b69f0dac147efadfed92f12712736\6.1.0-9-amd64\initrd.img-6.1.0-9-amd64
// root=zfs:AUTO fbcon=rotate:3 intel_iommu=on zfs.zfs_arc_max=12884901888 quiet module.sig_enforce=1
//
// for f in Boot000*; do { head -c 4 > /dev/null; cat; } < $f | sha1sum; echo $f; done | paste - -
// 6041dfca108fcdaff4b4247e449c8a9928de412c  -     Boot0001-8be4df61-93ca-11d2-aa0d-00e098032b8c
// 9ee7260155853ae8481fd8ae6f18ebc4391ddad0  -     Boot0002-8be4df61-93ca-11d2-aa0d-00e098032b8c
// 4467e2eb9e18e774708e2c7f1c9ded0233a5541f  -     Boot0003-8be4df61-93ca-11d2-aa0d-00e098032b8c
// fcb4a0c237990bb4563874f9a58c80839f8566c7  -     Boot0004-8be4df61-93ca-11d2-aa0d-00e098032b8c
// 01122d5bf9ba076e0b8fbb6ed168340e59e9bc1b  -     Boot0005-8be4df61-93ca-11d2-aa0d-00e098032b8c
//
// # hd -j 4 731b69f0dac147efadfed92f12712736-a8a9ad3a-f831-11ea-946d-674ccd7415cc
// 000004 00 00 62 72 65 61 6b 00 00 00 04 fc b4 a0 c2 37  >..break........7<
// 000014 99 0b b4 56 38 74 f9 a5 8c 80 83 9f 85 66 c7 36  >...V8t.......f.6<
// 000024 2e 31 2e 30 2d 37 2d 61 6d 64 36 34 00 00 2f 62  >.1.0-7-amd64../b<
// 000034 6f 6f 74 00 f5 20 5d 00 78 aa 0a f2 fb ce 65 bc  >oot.. ].x.....e.<
// 000044 81 d8 20 f2 1e 87 c3 a0 1a 00 fe 57 68 0e 12 c9  >.. ...à*...Wh...<
// 000054 f3 d4 92 46 72 7d ac 79 a8 27 92 a4 ac 45 00 00  >.Ԓ*Fr}.y.'...E..<
// 000064 05 01 12 2d 5b f9 ba 07 6e 0b 8f bb 6e d1 68 34  >...-[...n...n.h4<
// 000074 0e 59 e9 bc 1b 36 2e 31 2e 30 2d 39 2d 61 6d 64  >.Y...6.1.0-9-amd<
// 000084 36 34 00 00 2f 62 6f 6f 74 00 4e e9 6d 80 03 85  >64../boot.N.m...<
// 000094 4c 9d 9b 05 5f 15 b3 79 5b 72 66 75 71 9c 1a 00  >L..._..y[rfuq...<
// 0000a4 e2 ea 6e 42 b9 dc 1f 79 ee 95 6b 83 a9 14 99 3c  >..nB...y..k....<<
// 0000b4 17 48 04 ef 00 00 06 f3 76 3f 9c e6 88 b1 d1 b4  >.H......v?.戱*Ѵ*<
// 0000c4 b0 bb 11 b1 3a 01 98 ef b4 44 49 36 2e 31 2e 30  >....:....DI6.1.0<
// 0000d4 2d 37 2d 61 6d 64 36 34 00 62 72 65 61 6b 00 2f  >-7-amd64.break./<
// 0000e4 62 6f 6f 74 00 f5 20 5d 00 78 aa 0a f2 fb ce 65  >boot.. ].x.....e<
// 0000f4 bc 81 d8 20 f2 1e 87 c3 a0 1a 00 fe 57 68 0e 12  >... ...à*...Wh..<
// 000104 c9 f3 d4 92 46 72 7d ac 79 a8 27 92 a4 ac 45 00  >..Ԓ*Fr}.y.'...E.<
// 000114 00 07 19 bb 92 93 9c 9d 5c 3c de 2a bc 33 6f 29  >........\<.*.3o)<
// 000124 8d bd e1 4d d1 76 36 2e 31 2e 30 2d 39 2d 61 6d  >...M.v6.1.0-9-am<
// 000134 64 36 34 00 62 72 65 61 6b 00 2f 62 6f 6f 74 00  >d64.break./boot.<
// 000144 4e e9 6d 80 03 85 4c 9d 9b 05 5f 15 b3 79 5b 72  >N.m...L..._..y[r<
// 000154 66 75 71 9c 1a 00 e2 ea 6e 42 b9 dc 1f 79 ee 95  >fuq.....nB...y..<
// 000164 6b 83 a9 14 99 3c 17 48 04 ef 00                 >k....<.H...<
// 00016f


namespace {
	const klapki::config config{
	    .host            = "731b69f0dac147efadfed92f12712736"s,
	    .verbose         = false,
	    .commit          = true,
	    .esp             = "/boot/efi"sv,
	    .ops             = {},
	    .efi_root_raw    = "\\klapki"sv,
	    .wisdom_root_raw = "/tmp/" NAME "/"sv,
	};
	const std::string_view config_wisdom_description = "echo Debian GNU/Linux trixie/sid with Linux $1\n"
	                                                   "[ -n \"$2\" ] && echo \"($2)\"\n"
	                                                   ":";
	const std::string_view config_wisdom_cmdline     = "echo root=zfs:AUTO fbcon=rotate:3 intel_iommu=on zfs.zfs_arc_max=12884901888 quiet module.sig_enforce=1\n"
	                                                   "[ \"$2\" = break ] && echo rd.break=cmdline\n"
	                                                   ":";

	const klapki::state::state from_step_2{
	    klapki::state::boot_order_flat{std::shared_ptr<std::uint16_t[]>{new std::uint16_t[]{0x0005, 0x0004, 0x0001, 0x0002, 0x0003}}, 5},
	    {// Boot0001-8be4df61-93ca-11d2-aa0d-00e098032b8c
	     {0x0001, boot_entry_raw({0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x55, 0x00, 0x45, 0x00, 0x46, 0x00, 0x49, 0x00, 0x3a, 0x00,
	                              0x43, 0x00, 0x44, 0x00, 0x2f, 0x00, 0x44, 0x00, 0x56, 0x00, 0x44, 0x00, 0x20, 0x00, 0x44, 0x00, 0x72, 0x00, 0x69, 0x00,
	                              0x76, 0x00, 0x65, 0x00, 0x00, 0x00, 0x05, 0x01, 0x09, 0x00, 0x81, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0x04, 0x00})},
	     // Boot0002-8be4df61-93ca-11d2-aa0d-00e098032b8c
	     {0x0002, boot_entry_raw({0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x55, 0x00, 0x45, 0x00, 0x46, 0x00, 0x49,
	                              0x00, 0x3a, 0x00, 0x52, 0x00, 0x65, 0x00, 0x6d, 0x00, 0x6f, 0x00, 0x76, 0x00, 0x61, 0x00, 0x62, 0x00,
	                              0x6c, 0x00, 0x65, 0x00, 0x20, 0x00, 0x44, 0x00, 0x65, 0x00, 0x76, 0x00, 0x69, 0x00, 0x63, 0x00, 0x65,
	                              0x00, 0x00, 0x00, 0x05, 0x01, 0x09, 0x00, 0x82, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0x04, 0x00})},
	     // Boot0003-8be4df61-93ca-11d2-aa0d-00e098032b8c
	     {0x0003, boot_entry_raw({0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x55, 0x00, 0x45, 0x00, 0x46, 0x00, 0x49, 0x00, 0x3a, 0x00, 0x4e,
	                              0x00, 0x65, 0x00, 0x74, 0x00, 0x77, 0x00, 0x6f, 0x00, 0x72, 0x00, 0x6b, 0x00, 0x20, 0x00, 0x44, 0x00, 0x65, 0x00, 0x76, 0x00,
	                              0x69, 0x00, 0x63, 0x00, 0x65, 0x00, 0x00, 0x00, 0x05, 0x01, 0x09, 0x00, 0x83, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0x04, 0x00})},
	     // Boot0004-8be4df61-93ca-11d2-aa0d-00e098032b8c
	     {0x0004,
	      boot_entry_raw(
	          {0x07, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0xcc, 0x00, 0x44, 0x00, 0x65, 0x00, 0x62, 0x00, 0x69, 0x00, 0x61, 0x00, 0x6e, 0x00, 0x20, 0x00,
	           0x47, 0x00, 0x4e, 0x00, 0x55, 0x00, 0x2f, 0x00, 0x4c, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x75, 0x00, 0x78, 0x00, 0x20, 0x00, 0x74, 0x00, 0x72, 0x00,
	           0x69, 0x00, 0x78, 0x00, 0x69, 0x00, 0x65, 0x00, 0x2f, 0x00, 0x73, 0x00, 0x69, 0x00, 0x64, 0x00, 0x20, 0x00, 0x77, 0x00, 0x69, 0x00, 0x74, 0x00,
	           0x68, 0x00, 0x20, 0x00, 0x4c, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x75, 0x00, 0x78, 0x00, 0x20, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00,
	           0x30, 0x00, 0x2d, 0x00, 0x37, 0x00, 0x2d, 0x00, 0x61, 0x00, 0x6d, 0x00, 0x64, 0x00, 0x36, 0x00, 0x34, 0x00, 0x00, 0x00, 0x04, 0x01, 0x2a, 0x00,
	           0x01, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x51, 0x03, 0x52, 0x48,
	           0x2c, 0x6c, 0x17, 0x46, 0xa8, 0xd1, 0xf3, 0x53, 0xb7, 0x50, 0xef, 0x98, 0x02, 0x02, 0x04, 0x04, 0x9e, 0x00, 0x5c, 0x00, 0x4b, 0x00, 0x4c, 0x00,
	           0x41, 0x00, 0x50, 0x00, 0x4b, 0x00, 0x49, 0x00, 0x5c, 0x00, 0x37, 0x00, 0x33, 0x00, 0x31, 0x00, 0x42, 0x00, 0x36, 0x00, 0x39, 0x00, 0x46, 0x00,
	           0x30, 0x00, 0x44, 0x00, 0x41, 0x00, 0x43, 0x00, 0x31, 0x00, 0x34, 0x00, 0x37, 0x00, 0x45, 0x00, 0x46, 0x00, 0x41, 0x00, 0x44, 0x00, 0x46, 0x00,
	           0x45, 0x00, 0x44, 0x00, 0x39, 0x00, 0x32, 0x00, 0x46, 0x00, 0x31, 0x00, 0x32, 0x00, 0x37, 0x00, 0x31, 0x00, 0x32, 0x00, 0x37, 0x00, 0x33, 0x00,
	           0x36, 0x00, 0x5c, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x37, 0x00, 0x2d, 0x00, 0x41, 0x00, 0x4d, 0x00,
	           0x44, 0x00, 0x36, 0x00, 0x34, 0x00, 0x5c, 0x00, 0x56, 0x00, 0x4d, 0x00, 0x4c, 0x00, 0x49, 0x00, 0x4e, 0x00, 0x55, 0x00, 0x5a, 0x00, 0x2d, 0x00,
	           0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x37, 0x00, 0x2d, 0x00, 0x41, 0x00, 0x4d, 0x00, 0x44, 0x00, 0x36, 0x00,
	           0x34, 0x00, 0x00, 0x00, 0x7f, 0xff, 0x04, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x69, 0x00, 0x74, 0x00, 0x72, 0x00, 0x64, 0x00, 0x3d, 0x00, 0x5c, 0x00,
	           0x6b, 0x00, 0x6c, 0x00, 0x61, 0x00, 0x70, 0x00, 0x6b, 0x00, 0x69, 0x00, 0x5c, 0x00, 0x37, 0x00, 0x33, 0x00, 0x31, 0x00, 0x62, 0x00, 0x36, 0x00,
	           0x39, 0x00, 0x66, 0x00, 0x30, 0x00, 0x64, 0x00, 0x61, 0x00, 0x63, 0x00, 0x31, 0x00, 0x34, 0x00, 0x37, 0x00, 0x65, 0x00, 0x66, 0x00, 0x61, 0x00,
	           0x64, 0x00, 0x66, 0x00, 0x65, 0x00, 0x64, 0x00, 0x39, 0x00, 0x32, 0x00, 0x66, 0x00, 0x31, 0x00, 0x32, 0x00, 0x37, 0x00, 0x31, 0x00, 0x32, 0x00,
	           0x37, 0x00, 0x33, 0x00, 0x36, 0x00, 0x5c, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x37, 0x00, 0x2d, 0x00,
	           0x61, 0x00, 0x6d, 0x00, 0x64, 0x00, 0x36, 0x00, 0x34, 0x00, 0x5c, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x69, 0x00, 0x74, 0x00, 0x72, 0x00, 0x64, 0x00,
	           0x2e, 0x00, 0x69, 0x00, 0x6d, 0x00, 0x67, 0x00, 0x2d, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x37, 0x00,
	           0x2d, 0x00, 0x61, 0x00, 0x6d, 0x00, 0x64, 0x00, 0x36, 0x00, 0x34, 0x00, 0x20, 0x00, 0x72, 0x00, 0x6f, 0x00, 0x6f, 0x00, 0x74, 0x00, 0x3d, 0x00,
	           0x7a, 0x00, 0x66, 0x00, 0x73, 0x00, 0x3a, 0x00, 0x41, 0x00, 0x55, 0x00, 0x54, 0x00, 0x4f, 0x00, 0x20, 0x00, 0x66, 0x00, 0x62, 0x00, 0x63, 0x00,
	           0x6f, 0x00, 0x6e, 0x00, 0x3d, 0x00, 0x72, 0x00, 0x6f, 0x00, 0x74, 0x00, 0x61, 0x00, 0x74, 0x00, 0x65, 0x00, 0x3a, 0x00, 0x33, 0x00, 0x20, 0x00,
	           0x69, 0x00, 0x6e, 0x00, 0x74, 0x00, 0x65, 0x00, 0x6c, 0x00, 0x5f, 0x00, 0x69, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x6d, 0x00, 0x75, 0x00, 0x3d, 0x00,
	           0x6f, 0x00, 0x6e, 0x00, 0x20, 0x00, 0x7a, 0x00, 0x66, 0x00, 0x73, 0x00, 0x2e, 0x00, 0x7a, 0x00, 0x66, 0x00, 0x73, 0x00, 0x5f, 0x00, 0x61, 0x00,
	           0x72, 0x00, 0x63, 0x00, 0x5f, 0x00, 0x6d, 0x00, 0x61, 0x00, 0x78, 0x00, 0x3d, 0x00, 0x31, 0x00, 0x32, 0x00, 0x38, 0x00, 0x38, 0x00, 0x34, 0x00,
	           0x39, 0x00, 0x30, 0x00, 0x31, 0x00, 0x38, 0x00, 0x38, 0x00, 0x38, 0x00, 0x20, 0x00, 0x71, 0x00, 0x75, 0x00, 0x69, 0x00, 0x65, 0x00, 0x74, 0x00,
	           0x20, 0x00, 0x6d, 0x00, 0x6f, 0x00, 0x64, 0x00, 0x75, 0x00, 0x6c, 0x00, 0x65, 0x00, 0x2e, 0x00, 0x73, 0x00, 0x69, 0x00, 0x67, 0x00, 0x5f, 0x00,
	           0x65, 0x00, 0x6e, 0x00, 0x66, 0x00, 0x6f, 0x00, 0x72, 0x00, 0x63, 0x00, 0x65, 0x00, 0x3d, 0x00, 0x31, 0x00})},
	     // Boot0005-8be4df61-93ca-11d2-aa0d-00e098032b8c
	     {0x0005,
	      boot_entry_raw(
	          {0x07, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0xcc, 0x00, 0x44, 0x00, 0x65, 0x00, 0x62, 0x00, 0x69, 0x00, 0x61, 0x00, 0x6e, 0x00, 0x20, 0x00,
	           0x47, 0x00, 0x4e, 0x00, 0x55, 0x00, 0x2f, 0x00, 0x4c, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x75, 0x00, 0x78, 0x00, 0x20, 0x00, 0x74, 0x00, 0x72, 0x00,
	           0x69, 0x00, 0x78, 0x00, 0x69, 0x00, 0x65, 0x00, 0x2f, 0x00, 0x73, 0x00, 0x69, 0x00, 0x64, 0x00, 0x20, 0x00, 0x77, 0x00, 0x69, 0x00, 0x74, 0x00,
	           0x68, 0x00, 0x20, 0x00, 0x4c, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x75, 0x00, 0x78, 0x00, 0x20, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00,
	           0x30, 0x00, 0x2d, 0x00, 0x39, 0x00, 0x2d, 0x00, 0x61, 0x00, 0x6d, 0x00, 0x64, 0x00, 0x36, 0x00, 0x34, 0x00, 0x00, 0x00, 0x04, 0x01, 0x2a, 0x00,
	           0x01, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x51, 0x03, 0x52, 0x48,
	           0x2c, 0x6c, 0x17, 0x46, 0xa8, 0xd1, 0xf3, 0x53, 0xb7, 0x50, 0xef, 0x98, 0x02, 0x02, 0x04, 0x04, 0x9e, 0x00, 0x5c, 0x00, 0x4b, 0x00, 0x4c, 0x00,
	           0x41, 0x00, 0x50, 0x00, 0x4b, 0x00, 0x49, 0x00, 0x5c, 0x00, 0x37, 0x00, 0x33, 0x00, 0x31, 0x00, 0x42, 0x00, 0x36, 0x00, 0x39, 0x00, 0x46, 0x00,
	           0x30, 0x00, 0x44, 0x00, 0x41, 0x00, 0x43, 0x00, 0x31, 0x00, 0x34, 0x00, 0x37, 0x00, 0x45, 0x00, 0x46, 0x00, 0x41, 0x00, 0x44, 0x00, 0x46, 0x00,
	           0x45, 0x00, 0x44, 0x00, 0x39, 0x00, 0x32, 0x00, 0x46, 0x00, 0x31, 0x00, 0x32, 0x00, 0x37, 0x00, 0x31, 0x00, 0x32, 0x00, 0x37, 0x00, 0x33, 0x00,
	           0x36, 0x00, 0x5c, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x39, 0x00, 0x2d, 0x00, 0x41, 0x00, 0x4d, 0x00,
	           0x44, 0x00, 0x36, 0x00, 0x34, 0x00, 0x5c, 0x00, 0x56, 0x00, 0x4d, 0x00, 0x4c, 0x00, 0x49, 0x00, 0x4e, 0x00, 0x55, 0x00, 0x5a, 0x00, 0x2d, 0x00,
	           0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x39, 0x00, 0x2d, 0x00, 0x41, 0x00, 0x4d, 0x00, 0x44, 0x00, 0x36, 0x00,
	           0x34, 0x00, 0x00, 0x00, 0x7f, 0xff, 0x04, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x69, 0x00, 0x74, 0x00, 0x72, 0x00, 0x64, 0x00, 0x3d, 0x00, 0x5c, 0x00,
	           0x6b, 0x00, 0x6c, 0x00, 0x61, 0x00, 0x70, 0x00, 0x6b, 0x00, 0x69, 0x00, 0x5c, 0x00, 0x37, 0x00, 0x33, 0x00, 0x31, 0x00, 0x62, 0x00, 0x36, 0x00,
	           0x39, 0x00, 0x66, 0x00, 0x30, 0x00, 0x64, 0x00, 0x61, 0x00, 0x63, 0x00, 0x31, 0x00, 0x34, 0x00, 0x37, 0x00, 0x65, 0x00, 0x66, 0x00, 0x61, 0x00,
	           0x64, 0x00, 0x66, 0x00, 0x65, 0x00, 0x64, 0x00, 0x39, 0x00, 0x32, 0x00, 0x66, 0x00, 0x31, 0x00, 0x32, 0x00, 0x37, 0x00, 0x31, 0x00, 0x32, 0x00,
	           0x37, 0x00, 0x33, 0x00, 0x36, 0x00, 0x5c, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x39, 0x00, 0x2d, 0x00,
	           0x61, 0x00, 0x6d, 0x00, 0x64, 0x00, 0x36, 0x00, 0x34, 0x00, 0x5c, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x69, 0x00, 0x74, 0x00, 0x72, 0x00, 0x64, 0x00,
	           0x2e, 0x00, 0x69, 0x00, 0x6d, 0x00, 0x67, 0x00, 0x2d, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x39, 0x00,
	           0x2d, 0x00, 0x61, 0x00, 0x6d, 0x00, 0x64, 0x00, 0x36, 0x00, 0x34, 0x00, 0x20, 0x00, 0x72, 0x00, 0x6f, 0x00, 0x6f, 0x00, 0x74, 0x00, 0x3d, 0x00,
	           0x7a, 0x00, 0x66, 0x00, 0x73, 0x00, 0x3a, 0x00, 0x41, 0x00, 0x55, 0x00, 0x54, 0x00, 0x4f, 0x00, 0x20, 0x00, 0x66, 0x00, 0x62, 0x00, 0x63, 0x00,
	           0x6f, 0x00, 0x6e, 0x00, 0x3d, 0x00, 0x72, 0x00, 0x6f, 0x00, 0x74, 0x00, 0x61, 0x00, 0x74, 0x00, 0x65, 0x00, 0x3a, 0x00, 0x33, 0x00, 0x20, 0x00,
	           0x69, 0x00, 0x6e, 0x00, 0x74, 0x00, 0x65, 0x00, 0x6c, 0x00, 0x5f, 0x00, 0x69, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x6d, 0x00, 0x75, 0x00, 0x3d, 0x00,
	           0x6f, 0x00, 0x6e, 0x00, 0x20, 0x00, 0x7a, 0x00, 0x66, 0x00, 0x73, 0x00, 0x2e, 0x00, 0x7a, 0x00, 0x66, 0x00, 0x73, 0x00, 0x5f, 0x00, 0x61, 0x00,
	           0x72, 0x00, 0x63, 0x00, 0x5f, 0x00, 0x6d, 0x00, 0x61, 0x00, 0x78, 0x00, 0x3d, 0x00, 0x31, 0x00, 0x32, 0x00, 0x38, 0x00, 0x38, 0x00, 0x34, 0x00,
	           0x39, 0x00, 0x30, 0x00, 0x31, 0x00, 0x38, 0x00, 0x38, 0x00, 0x38, 0x00, 0x20, 0x00, 0x71, 0x00, 0x75, 0x00, 0x69, 0x00, 0x65, 0x00, 0x74, 0x00,
	           0x20, 0x00, 0x6d, 0x00, 0x6f, 0x00, 0x64, 0x00, 0x75, 0x00, 0x6c, 0x00, 0x65, 0x00, 0x2e, 0x00, 0x73, 0x00, 0x69, 0x00, 0x67, 0x00, 0x5f, 0x00,
	           0x65, 0x00, 0x6e, 0x00, 0x66, 0x00, 0x6f, 0x00, 0x72, 0x00, 0x63, 0x00, 0x65, 0x00, 0x3d, 0x00, 0x31, 0x00})}},

	    {.boot_position  = 0x0000,
	     .variants       = {"break"},
	     .wanted_entries = {
	         {.bootnum_hint     = 0x0004,
	          .load_option_sha  = {0xfc, 0xb4, 0xa0, 0xc2, 0x37, 0x99, 0xb, 0xb4, 0x56, 0x38, 0x74, 0xf9, 0xa5, 0x8c, 0x80, 0x83, 0x9f, 0x85, 0x66, 0xc7},
	          .version          = "6.1.0-7-amd64",
	          .variant          = "",
	          .kernel_dirname   = "/boot",
	          .kernel_image_sha = {0xf5, 0x20, 0x5d, 0x0, 0x78, 0xaa, 0xa, 0xf2, 0xfb, 0xce, 0x65, 0xbc, 0x81, 0xd8, 0x20, 0xf2, 0x1e, 0x87, 0xc3, 0xa0},
	          .initrd_dirnames  = {std::pair<klapki::state::nonbase_dirname_t, klapki::state::shaa_t>{
                /*.first  =*/std::nullopt,
                /*.second =*/{0xfe, 0x57, 0x68, 0xe, 0x12, 0xc9, 0xf3, 0xd4, 0x92, 0x46, 0x72, 0x7d, 0xac, 0x79, 0xa8, 0x27, 0x92, 0xa4, 0xac, 0x45}}}},
	         {.bootnum_hint     = 0x0005,
	          .load_option_sha  = {0x1, 0x12, 0x2d, 0x5b, 0xf9, 0xba, 0x7, 0x6e, 0xb, 0x8f, 0xbb, 0x6e, 0xd1, 0x68, 0x34, 0xe, 0x59, 0xe9, 0xbc, 0x1b},
	          .version          = "6.1.0-9-amd64",
	          .variant          = "",
	          .kernel_dirname   = "/boot",
	          .kernel_image_sha = {0x4e, 0xe9, 0x6d, 0x80, 0x3, 0x85, 0x4c, 0x9d, 0x9b, 0x5, 0x5f, 0x15, 0xb3, 0x79, 0x5b, 0x72, 0x66, 0x75, 0x71, 0x9c},
	          .initrd_dirnames  = {std::pair<klapki::state::nonbase_dirname_t, klapki::state::shaa_t>{
                /*.first  =*/std::nullopt,
                /*.second =*/{0xe2, 0xea, 0x6e, 0x42, 0xb9, 0xdc, 0x1f, 0x79, 0xee, 0x95, 0x6b, 0x83, 0xa9, 0x14, 0x99, 0x3c, 0x17, 0x48, 0x4, 0xef}}}},
	         {.bootnum_hint     = 0x0006,
	          .load_option_sha  = {0xf3, 0x76, 0x3f, 0x9c, 0xe6, 0x88, 0xb1, 0xd1, 0xb4, 0xb0, 0xbb, 0x11, 0xb1, 0x3a, 0x1, 0x98, 0xef, 0xb4, 0x44, 0x49},
	          .version          = "6.1.0-7-amd64",
	          .variant          = "break",
	          .kernel_dirname   = "/boot",
	          .kernel_image_sha = {0xf5, 0x20, 0x5d, 0x0, 0x78, 0xaa, 0xa, 0xf2, 0xfb, 0xce, 0x65, 0xbc, 0x81, 0xd8, 0x20, 0xf2, 0x1e, 0x87, 0xc3, 0xa0},
	          .initrd_dirnames  = {std::pair<klapki::state::nonbase_dirname_t, klapki::state::shaa_t>{
                /*.first  =*/std::nullopt,
                /*.second =*/{0xfe, 0x57, 0x68, 0xe, 0x12, 0xc9, 0xf3, 0xd4, 0x92, 0x46, 0x72, 0x7d, 0xac, 0x79, 0xa8, 0x27, 0x92, 0xa4, 0xac, 0x45}}}},
	         {.bootnum_hint     = 0x0007,
	          .load_option_sha  = {0x19, 0xbb, 0x92, 0x93, 0x9c, 0x9d, 0x5c, 0x3c, 0xde, 0x2a, 0xbc, 0x33, 0x6f, 0x29, 0x8d, 0xbd, 0xe1, 0x4d, 0xd1, 0x76},
	          .version          = "6.1.0-9-amd64",
	          .variant          = "break",
	          .kernel_dirname   = "/boot",
	          .kernel_image_sha = {0x4e, 0xe9, 0x6d, 0x80, 0x3, 0x85, 0x4c, 0x9d, 0x9b, 0x5, 0x5f, 0x15, 0xb3, 0x79, 0x5b, 0x72, 0x66, 0x75, 0x71, 0x9c},
	          .initrd_dirnames  = {std::pair<klapki::state::nonbase_dirname_t, klapki::state::shaa_t>{
                /*.first  =*/std::nullopt,
                /*.second =*/{0xe2, 0xea, 0x6e, 0x42, 0xb9, 0xdc, 0x1f, 0x79, 0xee, 0x95, 0x6b, 0x83, 0xa9, 0x14, 0x99, 0x3c, 0x17, 0x48, 0x4, 0xef}}}}}}};

	const klapki::state::state from_step_3 = [] {
		auto ret = from_step_2;

		ret.order = klapki::state::boot_order_structured{.foreign = {0x0001, 0x0002, 0x0003}, .ours = {0x0005, 0x0004}};

		std::memcpy(ret.entries[0x0001].load_option_sha, "\x60\x41\xdf\xca\x10\x8f\xcd\xaf\xf4\xb4\x24\x7e\x44\x9c\x8a\x99\x28\xde\x41\x2c", sizeof(klapki::sha_t));
		std::memcpy(ret.entries[0x0002].load_option_sha, "\x9e\xe7\x26\x01\x55\x85\x3a\xe8\x48\x1f\xd8\xae\x6f\x18\xeb\xc4\x39\x1d\xda\xd0", sizeof(klapki::sha_t));
		std::memcpy(ret.entries[0x0003].load_option_sha, "\x44\x67\xe2\xeb\x9e\x18\xe7\x74\x70\x8e\x2c\x7f\x1c\x9d\xed\x02\x33\xa5\x54\x1f", sizeof(klapki::sha_t));
		std::memcpy(ret.entries[0x0004].load_option_sha, "\xfc\xb4\xa0\xc2\x37\x99\x0b\xb4\x56\x38\x74\xf9\xa5\x8c\x80\x83\x9f\x85\x66\xc7", sizeof(klapki::sha_t));
		std::memcpy(ret.entries[0x0005].load_option_sha, "\x01\x12\x2d\x5b\xf9\xba\x07\x6e\x0b\x8f\xbb\x6e\xd1\x68\x34\x0e\x59\xe9\xbc\x1b", sizeof(klapki::sha_t));

		ret.statecfg.wanted_entries.erase(std::remove_if(std::begin(ret.statecfg.wanted_entries), std::end(ret.statecfg.wanted_entries),
		                                                 [](auto && we) { return we.bootnum_hint == 0x0006 || we.bootnum_hint == 0x0007; }),
		                                  std::end(ret.statecfg.wanted_entries));

		return ret;
	}();

	const klapki::context::context from_step_4{
	    .our_kernels = {
	        {0x0004,
	         {"Debian GNU/Linux trixie/sid with Linux 6.1.0-7-amd64",
	          R"(initrd=\klapki\731b69f0dac147efadfed92f12712736\6.1.0-7-amd64\initrd.img-6.1.0-7-amd64 root=zfs:AUTO fbcon=rotate:3 intel_iommu=on zfs.zfs_arc_max=12884901888 quiet module.sig_enforce=1)",
	          {R"(\KLAPKI\731B69F0DAC147EFADFED92F12712736\6.1.0-7-AMD64\)", R"(VMLINUZ-6.1.0-7-AMD64)"},
	          {{R"(\klapki\731b69f0dac147efadfed92f12712736\6.1.0-7-amd64\)", R"(initrd.img-6.1.0-7-amd64)"}}}},
	        {0x0005,
	         {"Debian GNU/Linux trixie/sid with Linux 6.1.0-9-amd64",
	          R"(initrd=\klapki\731b69f0dac147efadfed92f12712736\6.1.0-9-amd64\initrd.img-6.1.0-9-amd64 root=zfs:AUTO fbcon=rotate:3 intel_iommu=on zfs.zfs_arc_max=12884901888 quiet module.sig_enforce=1)",
	          {R"(\KLAPKI\731B69F0DAC147EFADFED92F12712736\6.1.0-9-AMD64\)", R"(VMLINUZ-6.1.0-9-AMD64)"},
	          {{R"(\klapki\731b69f0dac147efadfed92f12712736\6.1.0-9-amd64\)", R"(initrd.img-6.1.0-9-amd64)"}}}}}};

	const klapki::state::state from_step_5_state       = from_step_3;
	const klapki::context::context from_step_5_context = from_step_4;

	const klapki::state::state from_step_6_state = [] {
		auto ret = from_step_5_state;

		std::get<klapki::state::boot_order_structured>(ret.order).ours.emplace_back(0x0000);
		std::get<klapki::state::boot_order_structured>(ret.order).ours.emplace_back(0x0006);

		ret.entries[0x0000] = {.attributes = EFI_VARIABLE_NON_VOLATILE | EFI_VARIABLE_BOOTSERVICE_ACCESS | EFI_VARIABLE_RUNTIME_ACCESS};
		ret.entries[0x0006] = {.attributes = EFI_VARIABLE_NON_VOLATILE | EFI_VARIABLE_BOOTSERVICE_ACCESS | EFI_VARIABLE_RUNTIME_ACCESS};

		auto && went_0      = ret.statecfg.wanted_entries.emplace_back(from_step_2.statecfg.wanted_entries[2]);
		went_0.bootnum_hint = 0x0000;
		std::memset(went_0.load_option_sha, 0, sizeof(klapki::sha_t));

		auto && went_6      = ret.statecfg.wanted_entries.emplace_back(from_step_2.statecfg.wanted_entries[3]);
		went_6.bootnum_hint = 0x0006;
		std::memset(went_6.load_option_sha, 0, sizeof(klapki::sha_t));

		return ret;
	}();
	const klapki::context::context from_step_6_context = [] {
		auto ret = from_step_5_context;

		ret.our_kernels[0x0000]             = ret.our_kernels[0x0004];
		ret.our_kernels[0x0000].description = "";
		std::transform(std::begin(ret.our_kernels[0x0000].image_path.first), std::end(ret.our_kernels[0x0000].image_path.first),
		               std::begin(ret.our_kernels[0x0000].image_path.first), static_cast<int (*)(int)>(std::tolower));
		ret.our_kernels[0x0000].cmdline = "";

		ret.our_kernels[0x0006]             = ret.our_kernels[0x0005];
		ret.our_kernels[0x0006].description = "";
		std::transform(std::begin(ret.our_kernels[0x0006].image_path.first), std::end(ret.our_kernels[0x0006].image_path.first),
		               std::begin(ret.our_kernels[0x0006].image_path.first), static_cast<int (*)(int)>(std::tolower));
		ret.our_kernels[0x0006].cmdline = "";

		return ret;
	}();

	const klapki::state::state from_step_7_state       = from_step_6_state;
	const klapki::context::context from_step_7_context = from_step_6_context;

	const klapki::context::context from_step_8 = [] {
		auto ret = from_step_7_context;

		ret.our_kernels[0x0004].cmdline.erase(0, ret.our_kernels[0x0004].cmdline.find(' ') + 1);
		ret.our_kernels[0x0005].cmdline.erase(0, ret.our_kernels[0x0005].cmdline.find(' ') + 1);

		(ret.our_kernels[0x0000].description = ret.our_kernels[0x0004].description) += " (break)"sv;
		(ret.our_kernels[0x0000].cmdline = ret.our_kernels[0x0004].cmdline) += " rd.break=cmdline"sv;

		(ret.our_kernels[0x0006].description = ret.our_kernels[0x0005].description) += " (break)"sv;
		(ret.our_kernels[0x0006].cmdline = ret.our_kernels[0x0005].cmdline) += " rd.break=cmdline"sv;

		return ret;
	}();

	const klapki::state::state from_step_9_state = [] {
		auto ret = from_step_7_state;
		ret.order =
		    klapki::state::boot_order_flat{std::shared_ptr<std::uint16_t[]>{new std::uint16_t[]{0x0005, 0x0006, 0x0004, 0x0000, 0x0001, 0x0002, 0x0003}}, 7};

		ret.entries[0x0000].load_option     = std::shared_ptr<std::uint8_t[]>{new std::uint8_t[]{
        0x07, 0x00, 0x00, 0x00, 0xcc, 0x00, 0x44, 0x00, 0x65, 0x00, 0x62, 0x00, 0x69, 0x00, 0x61, 0x00, 0x6e, 0x00, 0x20, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x55,
        0x00, 0x2f, 0x00, 0x4c, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x75, 0x00, 0x78, 0x00, 0x20, 0x00, 0x74, 0x00, 0x72, 0x00, 0x69, 0x00, 0x78, 0x00, 0x69, 0x00,
        0x65, 0x00, 0x2f, 0x00, 0x73, 0x00, 0x69, 0x00, 0x64, 0x00, 0x20, 0x00, 0x77, 0x00, 0x69, 0x00, 0x74, 0x00, 0x68, 0x00, 0x20, 0x00, 0x4c, 0x00, 0x69,
        0x00, 0x6e, 0x00, 0x75, 0x00, 0x78, 0x00, 0x20, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x37, 0x00, 0x2d, 0x00,
        0x61, 0x00, 0x6d, 0x00, 0x64, 0x00, 0x36, 0x00, 0x34, 0x00, 0x20, 0x00, 0x28, 0x00, 0x62, 0x00, 0x72, 0x00, 0x65, 0x00, 0x61, 0x00, 0x6b, 0x00, 0x29,
        0x00, 0x00, 0x00, 0x04, 0x01, 0x2a, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x07, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x51, 0x03, 0x52, 0x48, 0x2c, 0x6c, 0x17, 0x46, 0xa8, 0xd1, 0xf3, 0x53, 0xb7, 0x50, 0xef, 0x98, 0x02, 0x02, 0x04, 0x04, 0x9e, 0x00, 0x5c,
        0x00, 0x4b, 0x00, 0x4c, 0x00, 0x41, 0x00, 0x50, 0x00, 0x4b, 0x00, 0x49, 0x00, 0x5c, 0x00, 0x37, 0x00, 0x33, 0x00, 0x31, 0x00, 0x42, 0x00, 0x36, 0x00,
        0x39, 0x00, 0x46, 0x00, 0x30, 0x00, 0x44, 0x00, 0x41, 0x00, 0x43, 0x00, 0x31, 0x00, 0x34, 0x00, 0x37, 0x00, 0x45, 0x00, 0x46, 0x00, 0x41, 0x00, 0x44,
        0x00, 0x46, 0x00, 0x45, 0x00, 0x44, 0x00, 0x39, 0x00, 0x32, 0x00, 0x46, 0x00, 0x31, 0x00, 0x32, 0x00, 0x37, 0x00, 0x31, 0x00, 0x32, 0x00, 0x37, 0x00,
        0x33, 0x00, 0x36, 0x00, 0x5c, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x37, 0x00, 0x2d, 0x00, 0x41, 0x00, 0x4d,
        0x00, 0x44, 0x00, 0x36, 0x00, 0x34, 0x00, 0x5c, 0x00, 0x56, 0x00, 0x4d, 0x00, 0x4c, 0x00, 0x49, 0x00, 0x4e, 0x00, 0x55, 0x00, 0x5a, 0x00, 0x2d, 0x00,
        0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x37, 0x00, 0x2d, 0x00, 0x41, 0x00, 0x4d, 0x00, 0x44, 0x00, 0x36, 0x00, 0x34,
        0x00, 0x00, 0x00, 0x7f, 0xff, 0x04, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x69, 0x00, 0x74, 0x00, 0x72, 0x00, 0x64, 0x00, 0x3d, 0x00, 0x5c, 0x00, 0x6b, 0x00,
        0x6c, 0x00, 0x61, 0x00, 0x70, 0x00, 0x6b, 0x00, 0x69, 0x00, 0x5c, 0x00, 0x37, 0x00, 0x33, 0x00, 0x31, 0x00, 0x62, 0x00, 0x36, 0x00, 0x39, 0x00, 0x66,
        0x00, 0x30, 0x00, 0x64, 0x00, 0x61, 0x00, 0x63, 0x00, 0x31, 0x00, 0x34, 0x00, 0x37, 0x00, 0x65, 0x00, 0x66, 0x00, 0x61, 0x00, 0x64, 0x00, 0x66, 0x00,
        0x65, 0x00, 0x64, 0x00, 0x39, 0x00, 0x32, 0x00, 0x66, 0x00, 0x31, 0x00, 0x32, 0x00, 0x37, 0x00, 0x31, 0x00, 0x32, 0x00, 0x37, 0x00, 0x33, 0x00, 0x36,
        0x00, 0x5c, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x37, 0x00, 0x2d, 0x00, 0x61, 0x00, 0x6d, 0x00, 0x64, 0x00,
        0x36, 0x00, 0x34, 0x00, 0x5c, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x69, 0x00, 0x74, 0x00, 0x72, 0x00, 0x64, 0x00, 0x2e, 0x00, 0x69, 0x00, 0x6d, 0x00, 0x67,
        0x00, 0x2d, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x37, 0x00, 0x2d, 0x00, 0x61, 0x00, 0x6d, 0x00, 0x64, 0x00,
        0x36, 0x00, 0x34, 0x00, 0x20, 0x00, 0x72, 0x00, 0x6f, 0x00, 0x6f, 0x00, 0x74, 0x00, 0x3d, 0x00, 0x7a, 0x00, 0x66, 0x00, 0x73, 0x00, 0x3a, 0x00, 0x41,
        0x00, 0x55, 0x00, 0x54, 0x00, 0x4f, 0x00, 0x20, 0x00, 0x66, 0x00, 0x62, 0x00, 0x63, 0x00, 0x6f, 0x00, 0x6e, 0x00, 0x3d, 0x00, 0x72, 0x00, 0x6f, 0x00,
        0x74, 0x00, 0x61, 0x00, 0x74, 0x00, 0x65, 0x00, 0x3a, 0x00, 0x33, 0x00, 0x20, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x74, 0x00, 0x65, 0x00, 0x6c, 0x00, 0x5f,
        0x00, 0x69, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x6d, 0x00, 0x75, 0x00, 0x3d, 0x00, 0x6f, 0x00, 0x6e, 0x00, 0x20, 0x00, 0x7a, 0x00, 0x66, 0x00, 0x73, 0x00,
        0x2e, 0x00, 0x7a, 0x00, 0x66, 0x00, 0x73, 0x00, 0x5f, 0x00, 0x61, 0x00, 0x72, 0x00, 0x63, 0x00, 0x5f, 0x00, 0x6d, 0x00, 0x61, 0x00, 0x78, 0x00, 0x3d,
        0x00, 0x31, 0x00, 0x32, 0x00, 0x38, 0x00, 0x38, 0x00, 0x34, 0x00, 0x39, 0x00, 0x30, 0x00, 0x31, 0x00, 0x38, 0x00, 0x38, 0x00, 0x38, 0x00, 0x20, 0x00,
        0x71, 0x00, 0x75, 0x00, 0x69, 0x00, 0x65, 0x00, 0x74, 0x00, 0x20, 0x00, 0x6d, 0x00, 0x6f, 0x00, 0x64, 0x00, 0x75, 0x00, 0x6c, 0x00, 0x65, 0x00, 0x2e,
        0x00, 0x73, 0x00, 0x69, 0x00, 0x67, 0x00, 0x5f, 0x00, 0x65, 0x00, 0x6e, 0x00, 0x66, 0x00, 0x6f, 0x00, 0x72, 0x00, 0x63, 0x00, 0x65, 0x00, 0x3d, 0x00,
        0x31, 0x00, 0x20, 0x00, 0x72, 0x00, 0x64, 0x00, 0x2e, 0x00, 0x62, 0x00, 0x72, 0x00, 0x65, 0x00, 0x61, 0x00, 0x6b, 0x00, 0x3d, 0x00, 0x63, 0x00, 0x6d,
        0x00, 0x64, 0x00, 0x6c, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x65, 0x00}};
		ret.entries[0x0000].load_option_len = 736;
		klapki::SHA1(ret.entries[0x0000].load_option.get(), ret.entries[0x0000].load_option_len, ret.entries[0x0000].load_option_sha);
		std::memcpy(ret.statecfg.wanted_entries[2].load_option_sha, ret.entries[0x0000].load_option_sha, sizeof(klapki::sha_t));

		ret.entries[0x0006].load_option     = std::shared_ptr<std::uint8_t[]>{new std::uint8_t[]{
        0x07, 0x00, 0x00, 0x00, 0xcc, 0x00, 0x44, 0x00, 0x65, 0x00, 0x62, 0x00, 0x69, 0x00, 0x61, 0x00, 0x6e, 0x00, 0x20, 0x00, 0x47, 0x00, 0x4e, 0x00, 0x55,
        0x00, 0x2f, 0x00, 0x4c, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x75, 0x00, 0x78, 0x00, 0x20, 0x00, 0x74, 0x00, 0x72, 0x00, 0x69, 0x00, 0x78, 0x00, 0x69, 0x00,
        0x65, 0x00, 0x2f, 0x00, 0x73, 0x00, 0x69, 0x00, 0x64, 0x00, 0x20, 0x00, 0x77, 0x00, 0x69, 0x00, 0x74, 0x00, 0x68, 0x00, 0x20, 0x00, 0x4c, 0x00, 0x69,
        0x00, 0x6e, 0x00, 0x75, 0x00, 0x78, 0x00, 0x20, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x39, 0x00, 0x2d, 0x00,
        0x61, 0x00, 0x6d, 0x00, 0x64, 0x00, 0x36, 0x00, 0x34, 0x00, 0x20, 0x00, 0x28, 0x00, 0x62, 0x00, 0x72, 0x00, 0x65, 0x00, 0x61, 0x00, 0x6b, 0x00, 0x29,
        0x00, 0x00, 0x00, 0x04, 0x01, 0x2a, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x07, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x51, 0x03, 0x52, 0x48, 0x2c, 0x6c, 0x17, 0x46, 0xa8, 0xd1, 0xf3, 0x53, 0xb7, 0x50, 0xef, 0x98, 0x02, 0x02, 0x04, 0x04, 0x9e, 0x00, 0x5c,
        0x00, 0x4b, 0x00, 0x4c, 0x00, 0x41, 0x00, 0x50, 0x00, 0x4b, 0x00, 0x49, 0x00, 0x5c, 0x00, 0x37, 0x00, 0x33, 0x00, 0x31, 0x00, 0x42, 0x00, 0x36, 0x00,
        0x39, 0x00, 0x46, 0x00, 0x30, 0x00, 0x44, 0x00, 0x41, 0x00, 0x43, 0x00, 0x31, 0x00, 0x34, 0x00, 0x37, 0x00, 0x45, 0x00, 0x46, 0x00, 0x41, 0x00, 0x44,
        0x00, 0x46, 0x00, 0x45, 0x00, 0x44, 0x00, 0x39, 0x00, 0x32, 0x00, 0x46, 0x00, 0x31, 0x00, 0x32, 0x00, 0x37, 0x00, 0x31, 0x00, 0x32, 0x00, 0x37, 0x00,
        0x33, 0x00, 0x36, 0x00, 0x5c, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x39, 0x00, 0x2d, 0x00, 0x41, 0x00, 0x4d,
        0x00, 0x44, 0x00, 0x36, 0x00, 0x34, 0x00, 0x5c, 0x00, 0x56, 0x00, 0x4d, 0x00, 0x4c, 0x00, 0x49, 0x00, 0x4e, 0x00, 0x55, 0x00, 0x5a, 0x00, 0x2d, 0x00,
        0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x39, 0x00, 0x2d, 0x00, 0x41, 0x00, 0x4d, 0x00, 0x44, 0x00, 0x36, 0x00, 0x34,
        0x00, 0x00, 0x00, 0x7f, 0xff, 0x04, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x69, 0x00, 0x74, 0x00, 0x72, 0x00, 0x64, 0x00, 0x3d, 0x00, 0x5c, 0x00, 0x6b, 0x00,
        0x6c, 0x00, 0x61, 0x00, 0x70, 0x00, 0x6b, 0x00, 0x69, 0x00, 0x5c, 0x00, 0x37, 0x00, 0x33, 0x00, 0x31, 0x00, 0x62, 0x00, 0x36, 0x00, 0x39, 0x00, 0x66,
        0x00, 0x30, 0x00, 0x64, 0x00, 0x61, 0x00, 0x63, 0x00, 0x31, 0x00, 0x34, 0x00, 0x37, 0x00, 0x65, 0x00, 0x66, 0x00, 0x61, 0x00, 0x64, 0x00, 0x66, 0x00,
        0x65, 0x00, 0x64, 0x00, 0x39, 0x00, 0x32, 0x00, 0x66, 0x00, 0x31, 0x00, 0x32, 0x00, 0x37, 0x00, 0x31, 0x00, 0x32, 0x00, 0x37, 0x00, 0x33, 0x00, 0x36,
        0x00, 0x5c, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x39, 0x00, 0x2d, 0x00, 0x61, 0x00, 0x6d, 0x00, 0x64, 0x00,
        0x36, 0x00, 0x34, 0x00, 0x5c, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x69, 0x00, 0x74, 0x00, 0x72, 0x00, 0x64, 0x00, 0x2e, 0x00, 0x69, 0x00, 0x6d, 0x00, 0x67,
        0x00, 0x2d, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x31, 0x00, 0x2e, 0x00, 0x30, 0x00, 0x2d, 0x00, 0x39, 0x00, 0x2d, 0x00, 0x61, 0x00, 0x6d, 0x00, 0x64, 0x00,
        0x36, 0x00, 0x34, 0x00, 0x20, 0x00, 0x72, 0x00, 0x6f, 0x00, 0x6f, 0x00, 0x74, 0x00, 0x3d, 0x00, 0x7a, 0x00, 0x66, 0x00, 0x73, 0x00, 0x3a, 0x00, 0x41,
        0x00, 0x55, 0x00, 0x54, 0x00, 0x4f, 0x00, 0x20, 0x00, 0x66, 0x00, 0x62, 0x00, 0x63, 0x00, 0x6f, 0x00, 0x6e, 0x00, 0x3d, 0x00, 0x72, 0x00, 0x6f, 0x00,
        0x74, 0x00, 0x61, 0x00, 0x74, 0x00, 0x65, 0x00, 0x3a, 0x00, 0x33, 0x00, 0x20, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x74, 0x00, 0x65, 0x00, 0x6c, 0x00, 0x5f,
        0x00, 0x69, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x6d, 0x00, 0x75, 0x00, 0x3d, 0x00, 0x6f, 0x00, 0x6e, 0x00, 0x20, 0x00, 0x7a, 0x00, 0x66, 0x00, 0x73, 0x00,
        0x2e, 0x00, 0x7a, 0x00, 0x66, 0x00, 0x73, 0x00, 0x5f, 0x00, 0x61, 0x00, 0x72, 0x00, 0x63, 0x00, 0x5f, 0x00, 0x6d, 0x00, 0x61, 0x00, 0x78, 0x00, 0x3d,
        0x00, 0x31, 0x00, 0x32, 0x00, 0x38, 0x00, 0x38, 0x00, 0x34, 0x00, 0x39, 0x00, 0x30, 0x00, 0x31, 0x00, 0x38, 0x00, 0x38, 0x00, 0x38, 0x00, 0x20, 0x00,
        0x71, 0x00, 0x75, 0x00, 0x69, 0x00, 0x65, 0x00, 0x74, 0x00, 0x20, 0x00, 0x6d, 0x00, 0x6f, 0x00, 0x64, 0x00, 0x75, 0x00, 0x6c, 0x00, 0x65, 0x00, 0x2e,
        0x00, 0x73, 0x00, 0x69, 0x00, 0x67, 0x00, 0x5f, 0x00, 0x65, 0x00, 0x6e, 0x00, 0x66, 0x00, 0x6f, 0x00, 0x72, 0x00, 0x63, 0x00, 0x65, 0x00, 0x3d, 0x00,
        0x31, 0x00, 0x20, 0x00, 0x72, 0x00, 0x64, 0x00, 0x2e, 0x00, 0x62, 0x00, 0x72, 0x00, 0x65, 0x00, 0x61, 0x00, 0x6b, 0x00, 0x3d, 0x00, 0x63, 0x00, 0x6d,
        0x00, 0x64, 0x00, 0x6c, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x65, 0x00}};
		ret.entries[0x0006].load_option_len = 736;
		klapki::SHA1(ret.entries[0x0006].load_option.get(), ret.entries[0x0006].load_option_len, ret.entries[0x0006].load_option_sha);
		std::memcpy(ret.statecfg.wanted_entries[3].load_option_sha, ret.entries[0x0006].load_option_sha, sizeof(klapki::sha_t));

		return ret;
	}();
	const klapki::context::context from_step_9_context = from_step_8;

	// https://en.wikipedia.org/wiki/SHA-1#Example_hashes
	std::vector<std::tuple<const char *, bool, std::string_view, std::string_view>> step_10_files{
	    {"/boot/vmlinuz-6.1.0-7-amd64", false, "6.1.0-7-amd64"sv, "The quick brown fox jumps over the lazy dog"sv},     //
	    {"/boot/initrd.img-6.1.0-7-amd64", false, "6.1.0-7-amd64"sv, "The quick brown fox jumps over the lazy cog"sv},  //
	    {"/boot/vmlinuz-6.1.0-9-amd64", true, "6.1.0-9-amd64"sv, "The quick brown fox jumps over the lazy hog"sv},      //
	    {"/boot/initrd.img-6.1.0-9-amd64", true, "6.1.0-9-amd64"sv, "The quick brown fox jumps over the lazy bog"sv}};
	const klapki::state::state from_step_10 = [] {
		auto ret = from_step_9_state;

		std::memcpy(ret.statecfg.wanted_entries[0].kernel_image_sha, "\x2f\xd4\xe1\xc6\x7a\x2d\x28\xfc\xed\x84\x9e\xe1\xbb\x76\xe7\x39\x1b\x93\xeb\x12",
		            sizeof(klapki::sha_t));
		std::memcpy(ret.statecfg.wanted_entries[2].kernel_image_sha, "\x2f\xd4\xe1\xc6\x7a\x2d\x28\xfc\xed\x84\x9e\xe1\xbb\x76\xe7\x39\x1b\x93\xeb\x12",
		            sizeof(klapki::sha_t));
		std::memcpy(ret.statecfg.wanted_entries[0].initrd_dirnames[0].second.data(),
		            "\xde\x9f\x2c\x7f\xd2\x5e\x1b\x3a\xfa\xd3\xe8\x5a\x0b\xd1\x7d\x9b\x10\x0d\xb4\xb3", sizeof(klapki::sha_t));
		std::memcpy(ret.statecfg.wanted_entries[2].initrd_dirnames[0].second.data(),
		            "\xde\x9f\x2c\x7f\xd2\x5e\x1b\x3a\xfa\xd3\xe8\x5a\x0b\xd1\x7d\x9b\x10\x0d\xb4\xb3", sizeof(klapki::sha_t));

		std::memcpy(ret.statecfg.wanted_entries[1].kernel_image_sha, "\x9b\xf3\x54\xa7\x35\x97\xf3\xe3\x02\xfb\x13\x37\xbb\x96\xd9\xb1\x60\x18\x1f\xba",
		            sizeof(klapki::sha_t));
		std::memcpy(ret.statecfg.wanted_entries[3].kernel_image_sha, "\x9b\xf3\x54\xa7\x35\x97\xf3\xe3\x02\xfb\x13\x37\xbb\x96\xd9\xb1\x60\x18\x1f\xba",
		            sizeof(klapki::sha_t));
		std::memcpy(ret.statecfg.wanted_entries[1].initrd_dirnames[0].second.data(),
		            "\xf5\x87\xdb\x11\xca\xd4\xd9\x50\x96\xff\xcd\x36\xc7\x64\xd3\xf4\x98\x6a\xc3\xf9", sizeof(klapki::sha_t));
		std::memcpy(ret.statecfg.wanted_entries[3].initrd_dirnames[0].second.data(),
		            "\xf5\x87\xdb\x11\xca\xd4\xd9\x50\x96\xff\xcd\x36\xc7\x64\xd3\xf4\x98\x6a\xc3\xf9", sizeof(klapki::sha_t));

		return ret;
	}();
}
