// SPDX-License-Identifier: 0BSD


#include "config.hpp"
#include "context.hpp"
#include "state.hpp"
#include "vore-file"
#include <doctest/doctest.h>
#include <time.h>


constexpr static bool operator<(const struct timespec & lhs, const struct timespec & rhs) {
	return lhs.tv_sec < rhs.tv_sec || (lhs.tv_sec == rhs.tv_sec && lhs.tv_nsec < rhs.tv_nsec);
}

constexpr static struct timespec operator-(struct timespec lhs, const struct timespec & rhs) {
	if(rhs.tv_nsec > lhs.tv_nsec) {
		lhs.tv_sec -= 1;
		lhs.tv_nsec += 1'000'000'000;
	}
	lhs.tv_nsec -= rhs.tv_nsec;
	lhs.tv_sec -= rhs.tv_sec;
	return lhs;
}

doctest::String toString(const struct timespec & value);
doctest::String toString(const struct timespec & value) {
	return fmt::format("{}.{:09}", value.tv_sec, value.tv_nsec).c_str();
}


using namespace std::literals;


#pragma GCC diagnostic ignored "-Wmissing-field-initializers"
static const klapki::config config{.wisdom_root_raw = "/tmp/wisdom_speed/"sv};

static const klapki::state::state state{.statecfg = {.wanted_entries = {{0, {}, "version0"s, "variant0"s},
                                                                        {1, {}, "version1"s, "variant1"s},
                                                                        {2, {}, "version2"s, "variant2"s},
                                                                        {3, {}, "version3"s, "variant3"s},
                                                                        {4, {}, "version4"s, "variant4"s},
                                                                        {5, {}, "version5"s, "variant5"s},
                                                                        {6, {}, "version6"s, "variant6"s},
                                                                        {7, {}, "version7"s, "variant7"s}}}};

static const decltype(klapki::context::context{}.our_kernels) out_kern{
    {0, {.description = "description version0 variant0"s, .cmdline = "cmdline version0 variant0"s}},
    {1, {.description = "description version1 variant1"s, .cmdline = "cmdline version1 variant1"s}},
    {2, {.description = "description version2 variant2"s, .cmdline = "cmdline version2 variant2"s}},
    {3, {.description = "description version3 variant3"s, .cmdline = "cmdline version3 variant3"s}},
    {4, {.description = "description version4 variant4"s, .cmdline = "cmdline version4 variant4"s}},
    {5, {.description = "description version5 variant5"s, .cmdline = "cmdline version5 variant5"s}},
    {6, {.description = "description version6 variant6"s, .cmdline = "cmdline version6 variant6"s}},
    {7, {.description = "description version7 variant7"s, .cmdline = "cmdline version7 variant7"s}}};


TEST_CASE("wisdom speed test") {
	REQUIRE(!mkdir(config.wisdom_root().data(), 0777));
	for(auto && bit : {"description"sv, "cmdline"sv}) {
		auto data = ("#!/bin/sh\nsleep 0.1\necho "s += bit) += " $*\n"sv;
		vore::file::fd out{(std::string{config.wisdom_root()} += bit).c_str(), O_WRONLY | O_CREAT | O_CLOEXEC, 0777};
		REQUIRE(write(out, data.data(), data.size()) == data.size());
	}

	struct timespec beg, end;
	clock_gettime(CLOCK_MONOTONIC, &beg);

	klapki::context::context ctx{.our_kernels = {{0, {}}, {1, {}}, {2, {}}, {3, {}}, {4, {}}, {5, {}}, {6, {}}, {7, {}}}};
	REQUIRE(!ctx.wisen(config, state));

	clock_gettime(CLOCK_MONOTONIC, &end);
	REQUIRE((end - beg) < timespec{.tv_nsec = 125'000'000});

	REQUIRE(ctx.our_kernels == out_kern);
}
