// SPDX-License-Identifier: MIT
/*! klapki::context::context::derive(): step 4
 * Extract output_state klapki::state::state into a kernel/initrd listing:
 * copy every entry found in the klapki state and accounted for in the boot entries (state::statecfg::wanted_entries)
 * into context::context::our_kernels:
 *   * parse the Boot1234 variable data from the corresponding state::entries key
 *   * parse the path from File in our HD(...)\File(...)\Entire format -> UCS-2->UTF-8 -> context::context::our_kernel::image_path ("\klapki\...\", "vml...")
 *     (reject all that fail parsing or aren't in our preferred format)
 *   * extract the description -> context::context::our_kernel::description ("Debian" )          (just informational at this point)
 *   * parse the optional data -> context::context::our_kernel::cmdline ("initrd=\... root=...") (just informational at this point)
 *   * tokenise context::context::our_kernel::cmdline
 *     -> take tokens starting with initrd= and parse them to be ("\klapki\...\", "initrd...")
 *        up to the amount listed in the corresponding state::statecfg::wanted_entries::initrd_paths
 *     -> context::context::our_kernel::initrds
 */


#include "config.hpp"
#include "context.hpp"
#include "context_detail.hpp"
#include "iconv.hpp"
#include "util.hpp"
#include "vore-token"
#include <fmt/format.h>
#include <string_view>
extern "C" {
#include <efiboot.h>
#include <efivar-dp.h>
}

using namespace std::literals;


namespace {
	/// iconv<"UCS-2", "UTF-8", std::uint16_t> but stop at first NUL and append to out
	void iconv_UCS2_UTF8_NUL(const std::uint16_t * chars, size_t maxlen, std::string & out) {
		klapki::iconv<"UCS-2", "UTF-8", std::uint16_t>(chars, std::find(chars, chars + maxlen, 0) - chars, [&](auto && segment) { out += segment; });
	}
}


klapki::context::context klapki::context::context::derive(const config & cfg, const state::state & output_state) {
	context ret{};

	for(auto && went : output_state.statecfg.wanted_entries)
		if(auto bent = output_state.entries.find(went.bootnum_hint); bent != std::end(output_state.entries)) {
			our_kernel kern{};

			const auto efi_opt        = reinterpret_cast<efi_load_option *>(bent->second.load_option.get());
			const ssize_t efi_opt_len = bent->second.load_option_len;
			if(!efi_loadopt_is_valid(efi_opt, efi_opt_len)) {
				fmt::print(stderr, "Entry {:04X} not a valid EFI Load Option. Abandoning.\n", went.bootnum_hint);
				continue;
			}

			const auto dp = efi_loadopt_path(efi_opt, efi_opt_len);
			if(dp->type != EFIDP_MEDIA_TYPE && dp->subtype != EFIDP_MEDIA_HD) {
				fmt::print(stderr, "Entry {:04X} not Media Device Path HD. Abandoning.\n", went.bootnum_hint);
				continue;
			}

			if(cfg.verbose)
				// device path ("HD(...)\File(...)")
				fmt::print(fgettext("Entry {:04X}: {}\n"), went.bootnum_hint, klapki::context::detail::fmt_devpath(dp, efi_opt_len));

			const efidp_data * file;
			switch(efidp_next_node(dp, &file)) {
				case -1:
					fmt::print(stderr, "Entry {:04X}: second (file) node: {}. Abandoning.\n", went.bootnum_hint, std::strerror(errno));
					continue;
				case 0:
					fmt::print(stderr, "Entry {:04X}: second (file) node doesn't exist. Abandoning.\n", went.bootnum_hint);
					continue;
				case 1: {
					if(file->type != EFIDP_MEDIA_TYPE && file->subtype != EFIDP_MEDIA_FILE) {
						fmt::print(stderr, "Entry {:04X} file not Media Device Path File. Abandoning.\n", went.bootnum_hint);
						continue;
					}

					iconv_UCS2_UTF8_NUL(file->file.name, file->length, kern.image_path.first);
					if(const auto last_backslash = kern.image_path.first.rfind('\\'); last_backslash != std::string::npos) {
						kern.image_path.second = kern.image_path.first.substr(last_backslash + 1);
						kern.image_path.first.erase(last_backslash + 1);
					} else {
						fmt::print(stderr, fgettext("Entry {:04X}: kernel image {}: no backslashes? Relating to root.\n"), went.bootnum_hint, kern.image_path.first);
						kern.image_path = {"\\", std::move(kern.image_path.first)};
					}
				} break;
				default:
					__builtin_unreachable();
			}

			const efidp_data * final_node;
			switch(efidp_next_node(file, &final_node)) {
				case -1:
					fmt::print(stderr, "Entry {:04X}: third (final) node: {}. Abandoning.\n", went.bootnum_hint, std::strerror(errno));
					continue;
				case 0:
					fmt::print(stderr, "Entry {:04X}: third (final) node doesn't exist. Abandoning.\n", went.bootnum_hint);
					continue;
				case 1: {
					if(final_node->type != EFIDP_END_TYPE || final_node->subtype != EFIDP_END_ENTIRE) {
						fmt::print(stderr, "Entry {:04X} final node not End Type End Entire. Abandoning.\n", went.bootnum_hint);
						continue;
					}
				} break;
				default:
					__builtin_unreachable();
			}

			if(cfg.verbose)
				// kernel image file [path, name]
				fmt::print(fgettext("Entry {:04X}: [\"{}\", \"{}\"]\n"), went.bootnum_hint, kern.image_path.first, kern.image_path.second);

			kern.description = reinterpret_cast<const char *>(efi_loadopt_desc(efi_opt, efi_opt_len));
			if(cfg.verbose)
				// description (cmdline)
				fmt::print(fgettext("Entry {:04X}: {}\n"), went.bootnum_hint, kern.description);

			unsigned char * opt_data;
			size_t opt_data_len;
			efi_loadopt_optional_data(efi_opt, efi_opt_len, &opt_data, &opt_data_len);
			iconv_UCS2_UTF8_NUL(reinterpret_cast<const std::uint16_t *>(opt_data), opt_data_len / sizeof(std::uint16_t), kern.cmdline);
			// Stored in cmdline for informational purposes only — we get the freshest one during wisen() anyway

			for(auto && tok : vore::soft_tokenise{kern.cmdline, " \f\n\r\t\v"sv}) {
				if(tok.starts_with("initrd="sv)) {
					if(kern.initrd_paths.size() == went.initrd_dirnames.size())
						// file path ("\abc\def\initrd.img-...")
						fmt::print(stderr, "Entry {:04X}: {} added by cmdline, ignoring\n", went.bootnum_hint, tok);
					else {
						auto initrd_path = tok.substr("initrd="sv.size());
						if(const auto last_backslash = initrd_path.rfind('\\'); last_backslash != std::string::npos)
							kern.initrd_paths.emplace_back(initrd_path.substr(0, last_backslash + 1), initrd_path.substr(last_backslash + 1));
						else {
							fmt::print(stderr, fgettext("Entry {:04X}: initrd={}: no backslashes? Relating to root.\n"), went.bootnum_hint, initrd_path);
							kern.initrd_paths.emplace_back("\\", initrd_path);
						}
					}
				}
			}

			ret.our_kernels.emplace(went.bootnum_hint, std::move(kern));
		} else
			throw __func__;  // unreachable, by this point all wanted entries match up to boot entries

	return ret;
}
