// SPDX-License-Identifier: MIT
/*! klapki::context::context::propagate(): step 6
 * allocate_kernel_variant() by copying a known variant
 * (find lowest unoccupied Boot1234 number,
 *  register it as "our" in state::order by appending it to state::boot_order_structured::ours,
 *  add it the config state::statecfg::wanted_entries with a dummy overall-sha and the new number
 *  add it to state::entries with the new number but blank data
 *  add it to context::context::our_kernels with EFI dir/image/initrds but no description or cmdline)
 * (or just editing the variant name for a just-freed entry) for new variants.
 * Remove and purge_allocations() Boot1234 entries from output state::state with entries unused due to variant count reduxion,
 * Only existing entries (state::statecfg::wanted_entries) are considered, context::fresh_kernels are untouched.
 */


#include "config.hpp"
#include "context.hpp"


std::optional<std::string> klapki::context::context::propagate(const config & cfg, state::state & state) {
	std::map<std::string, std::map<std::uint16_t, std::string>> version_variants;  // Can't be string_views because we write over wanted_entries later
	for(auto && went : state.statecfg.wanted_entries)
		version_variants[went.version][went.bootnum_hint] = went.variant;

	for(auto && [ver, vars] : version_variants) {
		const auto baseline_bootnum = vars.begin()->first;

		std::set<std::string_view> missing_variants;

		auto var_cbk = [&](const std::string_view & var) {
			if(const auto vitr = std::find_if(std::begin(vars), std::end(vars), [&](auto && vv) { return vv.second == var; }); vitr != std::end(vars))
				vars.extract(vitr);
			else
				missing_variants.emplace(var);
		};
		var_cbk("");
		for(auto && var : state.statecfg.variants)
			var_cbk(var);

		for(auto && mvar : missing_variants)
			if(!vars.empty()) {  // At this point vars has entries which didn't match any currently stated variant. Try to relocate missing variants onto them.
				const auto bootnum = vars.extract(vars.begin()).key();

				const auto went = std::find_if(std::begin(state.statecfg.wanted_entries), std::end(state.statecfg.wanted_entries),
				                               [&](auto && went) { return went.bootnum_hint == bootnum; });
				if(went == std::end(state.statecfg.wanted_entries))
					throw __func__;

				went->variant = mvar;
			} else {
				const auto baseline_went = std::find_if(std::begin(state.statecfg.wanted_entries), std::end(state.statecfg.wanted_entries),
				                                        [&](auto && went) { return went.bootnum_hint == baseline_bootnum; });
				if(baseline_went == std::end(state.statecfg.wanted_entries))
					throw __func__;
				const auto baseline_kern = this->our_kernels.find(baseline_bootnum);
				if(baseline_kern == std::end(this->our_kernels))
					throw __func__;

				this->allocate_kernel_variant(cfg, state, std::string{ver}, std::string{mvar}, baseline_went->kernel_dirname, baseline_went->kernel_image_sha,
				                              baseline_went->initrd_dirnames, baseline_kern->second.image_path.second, baseline_kern->second.initrd_paths);
			}

		if(!vars.empty())
			this->purge_allocations(state, [&](auto && skern) { return vars.find(skern.bootnum_hint) != std::end(vars); });
	}

	return {};
}
