// SPDX-License-Identifier: 0BSD


#pragma once

#include <iconv.h>
#include <string_view>


namespace klapki {
	namespace detail {
		template <std::size_t size>
		struct lit {
			constexpr lit(const char (&literal)[size]) { std::copy_n(literal, size, this->data); }
			char data[size];
		};
	}

	/// infallible conversions only! UCS-2/UTF-8 and vice versa are ok
	template <detail::lit from, detail::lit to, class char_t, class F>
	void iconv(const char_t * chars, std::size_t len, F && f) {
		static const iconv_t conv = [] {
			auto ret = iconv_open(to.data, from.data);
			if(ret == (iconv_t)-1)
				throw __func__;
			return ret;
		}();
		::iconv(conv, nullptr, 0, nullptr, 0);

		auto char_bytes = const_cast<char *>(reinterpret_cast<const char *>(chars));
		len *= sizeof(*chars);

		char out[128 * 3];  // max codepoint is 0xFFFF => 1110xxxx 10xxxxxx 10xxxxxx
		while(len) {
			auto out_i    = out;
			auto out_size = sizeof(out);

			auto ret = ::iconv(conv, &char_bytes, &len, &out_i, &out_size);
			f(std::string_view{out, sizeof(out) - out_size});

			if(ret == (std::size_t)-1)  // EILSEQ and EINVAL impossible, E2BIG only
				if(errno != E2BIG)
					throw __func__;
		}
	}
}
