// SPDX-License-Identifier: MIT
/*! klapki::state::state::commit(): step 11
 * Write to /sys/firmware/efi/efivars/:
 *   output state::statecfg != input state::statecfg? output state::statecfg -> state::state::stated_config::serialise() -> {host} {efi_guid_klapki}
 *   state::order (boot_order_flat)                                                                                      -> BootOrder
 *   for every Boot1234 entry known in the input state::state and the output state::state (::entries)
 *     in input but not in output: delete the variable
 *     in both:                    skip if they're the same, else
 *     in output but not in input: write the variable
 */


#include "efi.hpp"
#include "state.hpp"
#include <fmt/format.h>
extern "C" {
#pragma GCC diagnostic push
#if __clang__
#pragma GCC diagnostic ignored "-Wgnu-zero-variadic-macro-arguments"
#endif
#pragma GCC diagnostic ignored "-Wvariadic-macros"
#include <efivar.h>
#pragma GCC diagnostic pop
// Doesn't include efivar.h:
#include <efivar-guids.h>
}


#define TRY_OPT(...)              \
	if(auto err = __VA_ARGS__; err) \
		return err;


std::optional<std::string> klapki::state::state::commit(const char * us, bool may_compress_statecfg, const state & original_state) const {
	if(this->statecfg != original_state.statecfg) {
		fmt::print(fgettext("Updating state config\n"));

		auto statecfg_raw = this->statecfg.serialise(may_compress_statecfg);

		// This does fail if the state is too big. Dunno what to do about it
		if(efi_set_variable(klapki::efi_guid_klapki, us, statecfg_raw.data(), statecfg_raw.size(),
		                    EFI_VARIABLE_NON_VOLATILE | EFI_VARIABLE_BOOTSERVICE_ACCESS | EFI_VARIABLE_RUNTIME_ACCESS, 0600) < 0)
			return fmt::format(fngettext("Couldn't write new state config ({} byte): {}", "Couldn't write new state config ({} bytes): {}", statecfg_raw.size()),
			                   statecfg_raw.size(), std::strerror(errno));
	}


	TRY_OPT(std::visit(klapki::overload{
	                       [&](const klapki::state::boot_order_flat & now, const klapki::state::boot_order_flat & orig) -> std::optional<std::string> {
		                       if(now.order_cnt == orig.order_cnt && !std::memcmp(now.order.get(), orig.order.get(), now.order_cnt * sizeof(now.order[0])))
			                       return std::nullopt;

		                       fmt::print(fgettext("Updating boot order\n"));

		                       if(efi_set_variable(efi_guid_global, "BootOrder", reinterpret_cast<std::uint8_t *>(now.order.get()),
		                                           now.order_cnt * sizeof(now.order[0]),
		                                           EFI_VARIABLE_NON_VOLATILE | EFI_VARIABLE_BOOTSERVICE_ACCESS | EFI_VARIABLE_RUNTIME_ACCESS, 0600) < 0)
			                       return fmt::format(fgettext("Couldn't write new boot order: {}"), std::strerror(errno));
		                       return {};
	                       },
	                       [&](const auto &, const auto &) -> std::optional<std::string> {
		                       throw __func__;  // nothing can be done now. context::save() should've set this, and the original state is also flat
	                       },
	                   },
	                   this->order, original_state.order));


	const auto max_bootnum = std::max(this->entries.empty() ? static_cast<std::uint16_t>(0) : this->entries.rbegin()->first,
	                                  original_state.entries.empty() ? static_cast<std::uint16_t>(0) : original_state.entries.rbegin()->first);
	for(std::size_t i = 0; i <= max_bootnum; ++i) {
		const auto now  = this->entries.find(i);
		const auto orig = original_state.entries.find(i);

		if(now == std::end(this->entries) && orig == std::end(original_state.entries))  // gap
			continue;
		else if(now == std::end(this->entries) && orig != std::end(original_state.entries)) {  // deleted
			fmt::print(fgettext("Deleting entry {:04X}\n"), i);
			if(efi_del_variable(efi_guid_global, fmt::format("Boot{:04X}", i).c_str()) < 0)
				return fmt::format(fgettext("Couldn't delete entry for {:04X}: {}"), i, std::strerror(errno));
			continue;
		} else if(now != std::end(this->entries) && orig == std::end(original_state.entries))  // new
			;
		else if(now->second.load_option_len == orig->second.load_option_len &&
		        !std::memcmp(now->second.load_option.get(), orig->second.load_option.get(), now->second.load_option_len))  // exists in both, only write if changed
			continue;

		if(orig == std::end(original_state.entries))
			fmt::print(fgettext("Writing entry {:04X}\n"), i);
		else
			fmt::print(fgettext("Updating entry {:04X}\n"), i);
		if(efi_set_variable(efi_guid_global, fmt::format("Boot{:04X}", i).c_str(), now->second.load_option.get(), now->second.load_option_len,
		                    now->second.attributes, 0600) < 0)
			return fmt::format(fgettext("Couldn't write entry {:04X}: {}"), i, std::strerror(errno));
	}

	return {};
}
