// SPDX-License-Identifier: MIT


#include "config.hpp"
#include "context.hpp"
#include "state.hpp"
#include "vore-file"
#include <cassert>
#include <doctest/doctest.h>
#include <initializer_list>
#include <span>
#include <tuple>
#include <unistd.h>

using namespace std::literals;


namespace {
	// https://docs.kernel.org/filesystems/efivarfs.html:
	// When a content of an UEFI variable in /sys/firmware/efi/efivars is displayed, for example using "hexdump",
	// pay attention that the first 4 bytes of the output represent the UEFI variable attributes, in little-endian format.
	klapki::state::boot_entry boot_entry_raw(const std::initializer_list<std::uint8_t> & opt) {
		klapki::state::boot_entry ret{};

		ret.load_option = std::make_shared_for_overwrite<std::uint8_t[]>(opt.size() - 4);
		std::copy(std::begin(opt) + 4, std::end(opt), ret.load_option.get());

		ret.load_option_len = opt.size() - 4;

		std::copy_n(std::begin(opt), 4, reinterpret_cast<std::uint8_t *>(&ret.attributes));
		ret.attributes = le32toh(ret.attributes);

		return ret;
	}
}


namespace {
	extern const klapki::config config;
	extern const std::string_view config_wisdom_description;
	extern const std::string_view config_wisdom_cmdline;

	extern const klapki::state::state from_step_2;

	extern const klapki::state::state from_step_3;

	extern const klapki::context::context from_step_4;

	extern const klapki::state::state from_step_5_state;
	extern const klapki::context::context from_step_5_context;

	extern const klapki::state::state from_step_6_state;
	extern const klapki::context::context from_step_6_context;

	extern const klapki::state::state from_step_7_state;
	extern const klapki::context::context from_step_7_context;

	extern const klapki::context::context from_step_8;

	extern const klapki::state::state from_step_9_state;
	extern const klapki::context::context from_step_9_context;

	extern std::vector<std::tuple<const char *, bool, std::string_view, std::string_view>> step_10_files;
	extern const klapki::state::state from_step_10;
}


namespace klapki::state {
	static bool operator==(const boot_entry & lhs, const boot_entry & rhs) {
		return std::basic_string_view<std::uint8_t>{lhs.load_option.get(), lhs.load_option_len} ==
		           std::basic_string_view<std::uint8_t>{rhs.load_option.get(), rhs.load_option_len} &&  //
		       !std::memcmp(lhs.load_option_sha, rhs.load_option_sha, sizeof(klapki::sha_t)) &&         //
		       lhs.attributes == rhs.attributes;
	}
}

namespace {
	struct err {
		std::optional<std::string> op;
		operator bool() { return !this->op; }
	};
}

namespace doctest {
	template <>
	struct StringMaker<err> {
		static String convert(const err & e) { return e.op.value_or("{}"s).c_str(); }
	};
}

#define CONTEXT_EQ(yielded, from)                       \
	REQUIRE(yielded.our_kernels == from.our_kernels);     \
	REQUIRE(yielded.fresh_kernels == from.fresh_kernels); \
	REQUIRE(yielded.deleted_files == from.deleted_files);

#define STATE_EQ_STRUCTURED(yielded, from)                                                                                                              \
	REQUIRE(std::get<klapki::state::boot_order_structured>(yielded.order).ours == std::get<klapki::state::boot_order_structured>(from.order).ours);       \
	REQUIRE(std::get<klapki::state::boot_order_structured>(yielded.order).foreign == std::get<klapki::state::boot_order_structured>(from.order).foreign); \
	REQUIRE(yielded.entries == from.entries);                                                                                                             \
	REQUIRE(yielded.statecfg == from.statecfg);

TEST_CASE("klapki::context::resolve_state_context() " NAME " 2->3") {
	const auto yielded_step_3 = std::get<klapki::state::state>(klapki::context::resolve_state_context(config, from_step_2));
	STATE_EQ_STRUCTURED(yielded_step_3, from_step_3);
}

TEST_CASE("klapki::context::context::derive() " NAME " 3->4") {
	const auto yielded_step_4 = klapki::context::context::derive(config, from_step_3);
	CONTEXT_EQ(yielded_step_4, from_step_4);
}

TEST_CASE("klapki::op::execute() " NAME " 4->5") {
	auto yielded_step_5_state   = from_step_3;
	auto yielded_step_5_context = from_step_4;
	for(auto && op : config.ops)
		REQUIRE(err{klapki::op::execute(op, config, yielded_step_5_state, yielded_step_5_context)});

	STATE_EQ_STRUCTURED(yielded_step_5_state, from_step_5_state);

	CONTEXT_EQ(yielded_step_5_context, from_step_5_context);
}

TEST_CASE("klapki::context::context::propagate() " NAME " 5->6") {
	auto yielded_step_6_state   = from_step_5_state;
	auto yielded_step_6_context = from_step_5_context;
	REQUIRE(err{yielded_step_6_context.propagate(config, yielded_step_6_state)});

	STATE_EQ_STRUCTURED(yielded_step_6_state, from_step_6_state);

	CONTEXT_EQ(yielded_step_6_context, from_step_6_context);
}

TEST_CASE("klapki::context::context::age() " NAME " 6->7") {
	auto yielded_step_7_state   = from_step_6_state;
	auto yielded_step_7_context = from_step_6_context;
	REQUIRE(err{yielded_step_7_context.age(config, yielded_step_7_state)});

	STATE_EQ_STRUCTURED(yielded_step_7_state, from_step_7_state);

	CONTEXT_EQ(yielded_step_7_context, from_step_7_context);
}

namespace {
	void write_wisdom(const char * file, const std::string_view & data) {
		vore::file::fd out{file, O_CREAT | O_WRONLY | O_TRUNC | O_CLOEXEC, 0777};
		for(auto && dt : {"#!/bin/sh\n"sv, data, "\n"sv})
			REQUIRE(write(out, dt.data(), dt.size()) == dt.size());
	}
}

TEST_CASE("klapki::context::context::wisen() " NAME " 7->8") {
	REQUIRE(!mkdir(config.wisdom_root().data(), 0777));
	write_wisdom((std::string{config.wisdom_root()} += "description"sv).c_str(), config_wisdom_description);
	write_wisdom((std::string{config.wisdom_root()} += "cmdline"sv).c_str(), config_wisdom_cmdline);

	auto yielded_step_8 = from_step_7_context;
	REQUIRE(err{yielded_step_8.wisen(config, from_step_7_state)});

	CONTEXT_EQ(yielded_step_8, from_step_8);
}

#pragma GCC diagnostic ignored "-Wmissing-field-initializers"

namespace std {
	namespace {
		template <class E>
		bool operator==(const std::span<E> & lhs, const std::span<E> & rhs) {
			return lhs.size() == rhs.size() && std::equal(std::begin(lhs), std::end(lhs), std::begin(rhs));
		}
	}
}

namespace {
	efidp_data * HD_Entire = [] {
		efidp_data HD{.hd = {.header = {EFIDP_MEDIA_TYPE, EFIDP_MEDIA_HD, sizeof(efidp_hd)}}};
#ifdef INIT_HD
		INIT_HD
#else
		std::memcpy(HD.hd.signature, "faux klapki disk", sizeof(HD.hd.signature));
#endif
		const efidp_data Entire{.header = {EFIDP_END_TYPE, EFIDP_END_ENTIRE, sizeof(efidp_header)}};

		efidp_data *HD_ptr, *ret;
		assert(efidp_append_node(nullptr, &HD, &HD_ptr) == 0);
		assert(efidp_append_node(HD_ptr, &Entire, &ret) == 0);
		std::free(HD_ptr);
		return ret;
	}();
}

#define FLAT_BOOT_ORDER_EQ(yielded, from)                                                             \
	const auto & yielded##_order = std::get<klapki::state::boot_order_flat>(yielded.order);             \
	const auto & from##_order    = std::get<klapki::state::boot_order_flat>(from.order);                \
	std::span<std::uint16_t> yielded##_order_s{yielded##_order.order.get(), yielded##_order.order_cnt}; \
	std::span<std::uint16_t> from##_order_s{from##_order.order.get(), from##_order.order_cnt};          \
	REQUIRE(yielded##_order_s.size() == from##_order_s.size());                                         \
	REQUIRE(std::equal(std::begin(yielded##_order_s), std::end(yielded##_order_s), std::begin(from##_order_s)))

#define STATE_EQ_FLAT(yielded, from)        \
	FLAT_BOOT_ORDER_EQ(yielded, from);        \
	REQUIRE(yielded.entries == from.entries); \
	REQUIRE(yielded.statecfg == from.statecfg);

TEST_CASE("klapki::context::context::save() " NAME " 8->9") {
	auto yielded_step_9_state   = from_step_7_state;
	auto yielded_step_9_context = from_step_8;
	REQUIRE(err{yielded_step_9_context.save(config, yielded_step_9_state, HD_Entire)});

	STATE_EQ_FLAT(yielded_step_9_state, from_step_9_state);

	CONTEXT_EQ(yielded_step_9_context, from_step_9_context);
}

TEST_CASE("klapki::context::context::commit() " NAME " 9->10") {
	for(auto && [f, _, __, d] : step_10_files) {
		vore::file::fd fd{f, O_WRONLY | O_CREAT | O_CLOEXEC, 0666};
		REQUIRE(fd != -1);
		REQUIRE(write(fd, d.data(), d.size()) == d.size());
	}

	auto yielded_step_10 = from_step_9_state;
	REQUIRE(err{from_step_9_context.commit(config, yielded_step_10)});

	STATE_EQ_FLAT(yielded_step_10, from_step_10);

	for(auto && [f, upper, v, d] : step_10_files) {
		auto bn = std::strrchr(f, '/');
		std::string path{config.esp};
		auto addbig = [&](const std::string_view & what) {
			std::transform(std::begin(what), std::end(what), std::back_inserter(path), [&](char c) -> char {
				if(c == '\\')
					return '/';
				return upper ? std::toupper(c) : c;
			});
		};
		addbig(config.efi_root());
		(path += '/'), addbig(config.host);
		(path += '/'), addbig(v);
		(path += '/') += (bn ? bn + 1 : f);

		vore::file::fd fd{path.c_str(), O_RDONLY | O_CLOEXEC};
		REQUIRE(fd != -1);
		std::vector<char> dt;
		dt.resize(d.size());
		REQUIRE(read(fd, dt.data(), dt.size()) == dt.size());
		REQUIRE(!std::memcmp(dt.data(), d.data(), d.size()));
	}
}
