// SPDX-License-Identifier: MIT


#include "state.hpp"
#include "vore-file"
#include "vore-mmap"
#include <doctest/doctest.h>
#include <fmt/format.h>
#include <utility>

using namespace std::literals;


static const std::pair<const char *, klapki::state::stated_config> states[] =
    {{"just-bootplace", klapki::state::stated_config{0x0a01, {}, {}}},
     {"just-bootplace-but-0xFFFF", klapki::state::stated_config{0xFFFF, {}, {}}},
     {"just-kbase",
      klapki::state::stated_config{0x0b02,
                                   {},
                                   {{0x0102,
                                     {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13},
                                     "5.8.0-1-amd64",
                                     "",
                                     "/boot/",
                                     {0x13, 0x12, 0x11, 0x10, 0x0F, 0x0E, 0x0D, 0x0C, 0x0B, 0x0A, 0x09, 0x08, 0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00},
                                     {}}}}},
     {"one-initrd",
      klapki::state::stated_config{
          0x0b02,
          {"test"},
          {{0x0102,
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13},
            "5.8.0-1-amd64",
            "debug",
            "/boot/",
            {0x13, 0x12, 0x11, 0x10, 0x0F, 0x0E, 0x0D, 0x0C, 0x0B, 0x0A, 0x09, 0x08, 0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00},
            {{"/boot/initrds", {0x13, 0x12, 0x11, 0x10, 0x0F, 0x0E, 0x0D, 0x0C, 0x0B, 0x0A, 0x09, 0x08, 0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00}}}}}}},
     {"one-initrd-copy+two-initrds",
      klapki::state::stated_config{
          0x0b02,
          {"test", "2st"},
          {{0x0102,
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13},
            "5.8.0-1-amd64",
            "loud",
            "/boot/",
            {0x13, 0x12, 0x11, 0x10, 0x0F, 0x0E, 0x0D, 0x0C, 0x0B, 0x0A, 0x09, 0x08, 0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00},
            {{{}, {0x13, 0x12, 0x11, 0x10, 0x0F, 0x0E, 0x0D, 0x0C, 0x0B, 0x0A, 0x09, 0x08, 0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00}}}},
           {0x0102,
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13},
            "5.8.0-1-amd64",
            "quiet",
            "/boot/",
            {0x13, 0x12, 0x11, 0x10, 0x0F, 0x0E, 0x0D, 0x0C, 0x0B, 0x0A, 0x09, 0x08, 0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00},
            {
                {"/boot/initrds", {0x13, 0x12, 0x11, 0x10, 0x0F, 0x0E, 0x0D, 0x0C, 0x0B, 0x0A, 0x09, 0x08, 0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00}},
                {"/boot", {0x13, 0x12, 0x11, 0x10, 0x0F, 0x0E, 0x0D, 0x0C, 0x0B, 0x0A, 0x09, 0x08, 0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00}},
            }}}}},
     {"exempli-gratia",
      klapki::state::stated_config{
          0x0020,
          {"owo"},
          {{0x000B,
            {0x7C, 0x72, 0x2D, 0xF4, 0x1B, 0x78, 0xEF, 0xC7, 0xC4, 0x07, 0x20, 0xBA, 0x67, 0x54, 0xFC, 0x36, 0x6D, 0x8F, 0xE5, 0x39},
            "5.8.0-1-amd64",
            "",
            "/boot",
            {0x13, 0x12, 0x11, 0x10, 0x0F, 0x0E, 0x0D, 0x0C, 0x0B, 0x0A, 0x09, 0x08, 0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00},
            {{{}, {0x13, 0x12, 0x11, 0x10, 0x0F, 0x0E, 0x0D, 0x0C, 0x0B, 0x0A, 0x09, 0x08, 0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00}}}},
           {0x000D,
            {0xB0, 0x7C, 0x71, 0x35, 0x33, 0xEC, 0x0F, 0x2E, 0x21, 0x3C, 0xBF, 0xA2, 0x46, 0x1F, 0xD7, 0x25, 0xD5, 0x5D, 0xFE, 0xD4},
            "5.8.0-1-amd64",
            "owo",
            "/boot",
            {0x13, 0x12, 0x11, 0x10, 0x0F, 0x0E, 0x0D, 0x0C, 0x0B, 0x0A, 0x09, 0x08, 0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00},
            {{{}, {0x13, 0x12, 0x11, 0x10, 0x0F, 0x0E, 0x0D, 0x0C, 0x0B, 0x0A, 0x09, 0x08, 0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00}}}}}}},

     {"731b69f0dac147efadfed92f12712736-a8a9ad3a-f831-11ea-946d-674ccd7415cc",  // everything_babtop.hpp
      {.boot_position = 0x0000,
       .variants      = {"break"},
       .wanted_entries =
           {{.bootnum_hint     = 0x0004,
             .load_option_sha  = {0xfc, 0xb4, 0xa0, 0xc2, 0x37, 0x99, 0xb, 0xb4, 0x56, 0x38, 0x74, 0xf9, 0xa5, 0x8c, 0x80, 0x83, 0x9f, 0x85, 0x66, 0xc7},
             .version          = "6.1.0-7-amd64",
             .variant          = "",
             .kernel_dirname   = "/boot",
             .kernel_image_sha = {0xf5, 0x20, 0x5d, 0x0, 0x78, 0xaa, 0xa, 0xf2, 0xfb, 0xce, 0x65, 0xbc, 0x81, 0xd8, 0x20, 0xf2, 0x1e, 0x87, 0xc3, 0xa0},
             .initrd_dirnames  = {std::pair<klapki::state::nonbase_dirname_t, klapki::state::shaa_t>{
                 /*.first  =*/std::nullopt,
                 /*.second =*/{0xfe, 0x57, 0x68, 0xe, 0x12, 0xc9, 0xf3, 0xd4, 0x92, 0x46, 0x72, 0x7d, 0xac, 0x79, 0xa8, 0x27, 0x92, 0xa4, 0xac, 0x45}}}},
            {.bootnum_hint     = 0x0005,
             .load_option_sha  = {0x1, 0x12, 0x2d, 0x5b, 0xf9, 0xba, 0x7, 0x6e, 0xb, 0x8f, 0xbb, 0x6e, 0xd1, 0x68, 0x34, 0xe, 0x59, 0xe9, 0xbc, 0x1b},
             .version          = "6.1.0-9-amd64",
             .variant          = "",
             .kernel_dirname   = "/boot",
             .kernel_image_sha = {0x4e, 0xe9, 0x6d, 0x80, 0x3, 0x85, 0x4c, 0x9d, 0x9b, 0x5, 0x5f, 0x15, 0xb3, 0x79, 0x5b, 0x72, 0x66, 0x75, 0x71, 0x9c},
             .initrd_dirnames  = {std::pair<klapki::state::nonbase_dirname_t, klapki::state::shaa_t>{
                 /*.first  =*/std::nullopt,
                 /*.second =*/{0xe2, 0xea, 0x6e, 0x42, 0xb9, 0xdc, 0x1f, 0x79, 0xee, 0x95, 0x6b, 0x83, 0xa9, 0x14, 0x99, 0x3c, 0x17, 0x48, 0x4, 0xef}}}},
            {.bootnum_hint     = 0x0006,
             .load_option_sha  = {0xf3, 0x76, 0x3f, 0x9c, 0xe6, 0x88, 0xb1, 0xd1, 0xb4, 0xb0, 0xbb, 0x11, 0xb1, 0x3a, 0x1, 0x98, 0xef, 0xb4, 0x44, 0x49},
             .version          = "6.1.0-7-amd64",
             .variant          = "break",
             .kernel_dirname   = "/boot",
             .kernel_image_sha = {0xf5, 0x20, 0x5d, 0x0, 0x78, 0xaa, 0xa, 0xf2, 0xfb, 0xce, 0x65, 0xbc, 0x81, 0xd8, 0x20, 0xf2, 0x1e, 0x87, 0xc3, 0xa0},
             .initrd_dirnames  = {std::pair<klapki::state::nonbase_dirname_t, klapki::state::shaa_t>{
                 /*.first  =*/std::nullopt,
                 /*.second =*/{0xfe, 0x57, 0x68, 0xe, 0x12, 0xc9, 0xf3, 0xd4, 0x92, 0x46, 0x72, 0x7d, 0xac, 0x79, 0xa8, 0x27, 0x92, 0xa4, 0xac, 0x45}}}},
            {.bootnum_hint     = 0x0007,
             .load_option_sha  = {0x19, 0xbb, 0x92, 0x93, 0x9c, 0x9d, 0x5c, 0x3c, 0xde, 0x2a, 0xbc, 0x33, 0x6f, 0x29, 0x8d, 0xbd, 0xe1, 0x4d, 0xd1, 0x76},
             .version          = "6.1.0-9-amd64",
             .variant          = "break",
             .kernel_dirname   = "/boot",
             .kernel_image_sha = {0x4e, 0xe9, 0x6d, 0x80, 0x3, 0x85, 0x4c, 0x9d, 0x9b, 0x5, 0x5f, 0x15, 0xb3, 0x79, 0x5b, 0x72, 0x66, 0x75, 0x71, 0x9c},
             .initrd_dirnames  = {std::pair<klapki::state::nonbase_dirname_t, klapki::state::shaa_t>{
                 /*.first  =*/std::nullopt,
                 /*.second =*/{0xe2, 0xea, 0x6e, 0x42, 0xb9, 0xdc, 0x1f, 0x79, 0xee, 0x95, 0x6b, 0x83, 0xa9, 0x14, 0x99, 0x3c, 0x17, 0x48, 0x4, 0xef}}}}}}},
     {"a2b398a10f5f4af99258999e14093599-a8a9ad3a-f831-11ea-946d-674ccd7415cc",  // everything_rozbian.hpp
      {.boot_position  = 0x0000,
       .variants       = {},
       .wanted_entries = {klapki::state::stated_config_entry{
           .bootnum_hint     = 0x0006,
           .load_option_sha  = {0x99, 0x18, 0x3b, 0x20, 0x31, 0x5e, 0xb0, 0xe3, 0x8, 0xd2, 0x6a, 0xd9, 0x58, 0xd, 0x8d, 0x31, 0x87, 0xdc, 0xdb, 0x51},
           .version          = "6.4.0-1-amd64",
           .variant          = "",
           .kernel_dirname   = "/boot",
           .kernel_image_sha = {0xd, 0xb6, 0xa6, 0x5b, 0x68, 0x9, 0xa3, 0xad, 0xe5, 0x0, 0xf3, 0xf5, 0x5c, 0xbf, 0x93, 0x17, 0x7b, 0x6b, 0xb9, 0xcc},
           .initrd_dirnames  = {std::pair<klapki::state::nonbase_dirname_t, klapki::state::shaa_t>{
               /*.first  =*/std::nullopt,
               /*.second =*/{0x9a, 0xbb, 0x93, 0xc8, 0xe5, 0xf7, 0x6b, 0xf3, 0xf4, 0x23, 0x7c, 0x43, 0xbd, 0xc0, 0x24, 0x31, 0x21, 0xc7, 0x6f, 0xec}}}}}}},
     {"sr-a8a9ad3a-f831-11ea-946d-674ccd7415cc-result",  // everything_sr.hpp, final statecfg
      {.boot_position  = 0x0004,
       .variants       = {},
       .wanted_entries = {
           {.bootnum_hint     = 0x000A,
            .load_option_sha  = {0xfc, 0xfb, 0x67, 0xbc, 0x32, 0xc2, 0xbd, 0xce, 0x8e, 0xc5, 0x24, 0x3a, 0x70, 0xb7, 0x9, 0xe3, 0xf8, 0x9d, 0x2c, 0x60},
            .version          = "version",
            .variant          = "",
            .kernel_dirname   = "/",
            .kernel_image_sha = {0xa9, 0x99, 0x3e, 0x36, 0x47, 0x6, 0x81, 0x6a, 0xba, 0x3e, 0x25, 0x71, 0x78, 0x50, 0xc2, 0x6c, 0x9c, 0xd0, 0xd8, 0x9d},
            .initrd_dirnames  = {{/*.first =*/"/boot", /*.second =*/{0x84, 0x98, 0x3e, 0x44, 0x1c, 0x3b, 0xd2, 0x6e, 0xba, 0xae,
                                                                     0x4a, 0xa1, 0xf9, 0x51, 0x29, 0xe5, 0xe5, 0x46, 0x70, 0xf1}},
                                 {/*.first =*/std::nullopt, /*.second =*/{0xa4, 0x9b, 0x24, 0x46, 0xa0, 0x2c, 0x64, 0x5b, 0xf4, 0x19,
                                                                          0xf9, 0x95, 0xb6, 0x70, 0x91, 0x25, 0x3a, 0x4,  0xa2, 0x59}},
                                 {/*.first =*/"/tmp", /*.second =*/{0x34, 0xaa, 0x97, 0x3c, 0xd4, 0xc4, 0xda, 0xa4, 0xf6, 0x1e,
                                                                    0xeb, 0x2b, 0xdb, 0xad, 0x27, 0x31, 0x65, 0x34, 0x1,  0x6f}}}}}}}};


TEST_CASE("klapki::state::stated_config::parse() ENODATA") {
	klapki::state::stated_config clean{};
	clean.boot_position = __LINE__;
	auto cclean         = clean;

	klapki::state::stated_config::parse(clean, true, nullptr, 0);
	REQUIRE(clean == cclean);

	klapki::state::stated_config::parse(clean, true, nullptr, 1);
	REQUIRE(clean == cclean);
}

template <class F>
void foreach_nogz(F && f) {
	for(auto && [name, cfg] : states)
		SUBCASE(name) {
			vore::file::fd inf{("test-data/state_parse/"s += name).c_str(), O_RDONLY | O_CLOEXEC};
			REQUIRE(inf != -1);
			struct stat sb;
			fstat(inf, &sb);
			vore::file::mapping indata{nullptr, static_cast<std::size_t>(sb.st_size), PROT_READ, MAP_PRIVATE, inf};
			REQUIRE(indata);

			f(cfg, *indata);
		}
}

TEST_CASE("klapki::state::stated_config::parse() okay (no gzip)") {
	foreach_nogz([](auto && cfg, auto && indata) {
		klapki::state::stated_config incfg{};
		klapki::state::stated_config::parse(incfg, true, indata.data(), indata.size());

		REQUIRE(incfg == cfg);
	});
}

TEST_CASE("klapki::state::stated_config::serialise() okay (no gzip)") {
	foreach_nogz([](auto && cfg, auto && indata) {
		auto serialised = cfg.serialise(false);
		REQUIRE(std::basic_string_view{serialised.data(), serialised.size()} == indata);
	});
}


#if !KLAPKI_NO_ZLIB
TEST_CASE("klapki::state::stated_config::parse() okay (yes gzip)") {
	for(auto && [name, cfg] : states)
		if(vore::file::fd inf{(("test-data/state_parse/"s += name) += ".Z"sv).c_str(), O_RDONLY | O_CLOEXEC}; inf != -1)
			SUBCASE(name) {
				struct stat sb;
				fstat(inf, &sb);
				vore::file::mapping indata{nullptr, static_cast<std::size_t>(sb.st_size), PROT_READ, MAP_PRIVATE, inf};
				REQUIRE(indata);

				klapki::state::stated_config incfg{};
				klapki::state::stated_config::parse(incfg, true, indata->data(), indata->size());

				REQUIRE(incfg == cfg);
			}
}

TEST_CASE("klapki::state::stated_config::serialise() okay (yes gzip)") {
	for(auto && [name, cfg] : states)
		SUBCASE(name) {
			auto serialised = cfg.serialise(true);

			vore::file::fd inf{(("test-data/state_parse/"s += name) += ".Z"sv).c_str(), O_RDONLY | O_CLOEXEC};
			if(inf == -1 && errno == ENOENT)  // no .Z variant, shouldn't've gotten gzipped (<=> be the same as the base variant)
				inf = {("test-data/state_parse/"s += name).c_str(), O_RDONLY | O_CLOEXEC};
			REQUIRE(inf != -1);

			struct stat sb;
			fstat(inf, &sb);
			vore::file::mapping indata{nullptr, static_cast<std::size_t>(sb.st_size), PROT_READ, MAP_PRIVATE, inf};
			REQUIRE(indata);

			REQUIRE(std::basic_string_view{serialised.data(), serialised.size()} == *indata);
		}
}
#endif
