// SPDX-License-Identifier: MIT


#pragma once


#include "ops.hpp"
#include <libintl.h>
#include <string>
#include <string_view>
#include <variant>
#include <vector>


namespace klapki {
	struct config {
		std::string host;
		constexpr std::string_view wisdom_root() const noexcept;
		bool verbose;
		bool commit;
		bool compress;
		/// The first existing-mount-point of /boot/efi, /efi, or /boot.
		std::string_view esp;
		/// \klapki
		constexpr std::string_view efi_root() const noexcept;
		std::vector<ops::op_t> ops;

		std::variant<std::string_view, std::string> efi_root_raw;
		std::variant<std::string_view, std::string> wisdom_root_raw;

		static std::variant<config, std::string> read(int argc, char * const * argv);
	};
}


constexpr std::string_view klapki::config::efi_root() const noexcept {
	return std::visit([](auto && w) { return std::string_view{w}; }, this->efi_root_raw);
}

constexpr std::string_view klapki::config::wisdom_root() const noexcept {
	return std::visit([](auto && w) { return std::string_view{w}; }, this->wisdom_root_raw);
}

template <>
struct fmt::formatter<klapki::config> {
	constexpr auto parse(format_parse_context & ctx) { return ctx.begin(); }

	template <typename FormatContext>
	auto format(const klapki::config & cfg, FormatContext & ctx) {
		// hostname, /etc/klapki, true|false, true|false, true|false, /boot/efi, \klapki, ...
		return format_to(ctx.out(), fgettext("config{{ host: \"{}\", wisdom: \"{}\", verbose: {}, commit: {}, compress: {}, ESP: {}, ESP root: {}, ops: [{}] }}"),
		                 cfg.host, cfg.wisdom_root(), cfg.verbose, cfg.commit, cfg.compress, cfg.esp, cfg.efi_root(), klapki::mops{cfg.ops});
	}
};
