// SPDX-License-Identifier: MIT


#pragma once


#include "state.hpp"
#include <algorithm>
#include <fmt/format.h>
#include <map>
#include <set>
#include <string_view>
#include <variant>
#include <vector>
extern "C" {
#include <efiboot.h>
#include <efivar-dp.h>
}

namespace klapki {
	struct config;
}


namespace klapki::context {
	struct our_kernel {
		std::string description;
		std::string cmdline;

		std::pair<std::string, std::string> image_path;                              // path in ESP, basename
		std::vector<std::pair<state::nonbase_dirname_t, std::string>> initrd_paths;  // path in ESP, basename
	};

	struct fresh_kernel {
		std::string_view version;
		std::pair<std::string_view, std::string_view> image;
		std::vector<std::pair<std::string_view, std::string_view>> initrds;
	};

	struct context {
		std::map<std::uint16_t, our_kernel> our_kernels;
		std::vector<fresh_kernel> fresh_kernels;

		std::set<std::pair<std::string, std::string>> deleted_files;  // path in ESP, basename

		std::optional<std::string> allocate_kernel_variant(const config & cfg, state::state & state, std::string version, std::string var, std::string image_dir,
		                                                   sha_t & image_sha, std::vector<std::pair<state::nonbase_dirname_t, state::shaa_t>> initrd_dirs,
		                                                   std::string image_basename, std::vector<std::pair<state::nonbase_dirname_t, std::string>> initrd_paths);
		template <class F>
		void purge_allocations(state::state & state, F && pred);
		void purge_allocations_impl(state::state & state, const state::stated_config_entry & skern);

		std::optional<std::string> propagate(const config & cfg, state::state & state);
		std::optional<std::string> age(const config & cfg, state::state & state);
		std::optional<std::string> wisen(const config & cfg, const state::state & state);
		std::optional<std::string> save(const config & cfg, state::state & state, efidp_data * _test_esp_devpath_override = nullptr);
		std::optional<std::string> commit(const config & cfg, state::state & state) const;  // Only updates file SHAs in state

		static context derive(const config & cfg, const state::state & output_state);
	};

	std::variant<state::state, std::string> resolve_state_context(const config & cfg, const state::state & input_state);


	// common test support
	bool operator==(const our_kernel & lhs, const our_kernel & rhs);
	bool operator==(const fresh_kernel & lhs, const fresh_kernel & rhs);
}


template <class F>
void klapki::context::context::purge_allocations(state::state & state, F && pred) {
	state.statecfg.wanted_entries.erase(std::remove_if(std::begin(state.statecfg.wanted_entries), std::end(state.statecfg.wanted_entries),
	                                                   [&](auto && skern) {
		                                                   if(!pred(skern))
			                                                   return false;

		                                                   this->purge_allocations_impl(state, skern);
		                                                   return true;
	                                                   }),
	                                    std::end(state.statecfg.wanted_entries));
}


template <>
struct fmt::formatter<klapki::context::context> {
	constexpr auto parse(format_parse_context & ctx) { return ctx.begin(); }

	template <typename FormatContext>
	auto format(const klapki::context::context & context, FormatContext & ctx) {
		auto out = ctx.out();

		out = format_to(out, fngettext("{} of our kernels: [", "{} of our kernels: [", context.our_kernels.size()), context.our_kernels.size());

		bool first = true;
		for(auto && el : context.our_kernels) {
			if(first) {
				*out++ = '\n';
				first  = false;
			}

			out = format_to(out, fgettext("  {:04X}: {}\n"), el.first, el.second);
		}


		out = format_to(out, fgettext("]\n"
		                              "Fresh kernels: ["));

		first = true;
		for(auto && el : context.fresh_kernels) {
			if(first) {
				*out++ = '\n';
				first  = false;
			}

			out = format_to(out, "  {}\n", el);
		}

		out = format_to(out, fgettext("]\n"
		                              "Deleted files: ["));

		first = true;
		for(auto && el : context.deleted_files) {
			if(first) {
				*out++ = '\n';
				first  = false;
			}

			out = format_to(out, "  (\"{}\", \"{}\")\n", el.first, el.second);
		}

		out = format_to(out, "]");

		return out;
	}
};

template <>
struct fmt::formatter<klapki::context::our_kernel> {
	constexpr auto parse(format_parse_context & ctx) { return ctx.begin(); }

	template <typename FormatContext>
	auto format(const klapki::context::our_kernel & kern, FormatContext & ctx) {
		auto out = ctx.out();

		out = format_to(out, "{{ \"{}\", \"{}\", (\"{}\", \"{}\"), [", kern.description, kern.cmdline, kern.image_path.first, kern.image_path.second);

		bool first = true;
		for(auto && el : kern.initrd_paths) {
			if(!first)
				out = format_to(out, ", ");
			else
				first = false;

			*out++ = '(';
			if(el.first)
				*out++ = '\"';
			out = format_to(out, "{}", el.first ? std::string_view{*el.first} : "←");
			if(el.first)
				*out++ = '\"';
			out = format_to(out, ", \"{}\")", el.second);
		}

		out = format_to(out, "] }}");

		return out;
	}
};

template <>
struct fmt::formatter<klapki::context::fresh_kernel> {
	constexpr auto parse(format_parse_context & ctx) { return ctx.begin(); }

	template <typename FormatContext>
	auto format(const klapki::context::fresh_kernel & kern, FormatContext & ctx) {
		auto out = ctx.out();

		out = format_to(out, "{{ \"{}\", (\"{}\", \"{}\"), [", kern.version, kern.image.first, kern.image.second);

		bool first = true;
		for(auto && el : kern.initrds) {
			if(!first)
				out = format_to(out, ", ");
			else
				first = false;

			out = format_to(out, "(\"{}\", \"{}\")", el.first, el.second);
		}

		out = format_to(out, "] }}");

		return out;
	}
};
