// SPDX-License-Identifier: MIT
/*! klapki::context::context::age(): step 7
 * Feed context::fresh_kernels with absolute-then-deduplicated paths into allocate_kernel_variant() with all-zero file SHAs.
 */


#include "config.hpp"
#include "context.hpp"
#include "context_detail.hpp"
#include <algorithm>
#include <memory>
#include <stdlib.h>
extern "C" {
#pragma GCC diagnostic push
#if __clang__
#pragma GCC diagnostic ignored "-Wgnu-zero-variadic-macro-arguments"
#endif
#pragma GCC diagnostic ignored "-Wvariadic-macros"
#include <efivar.h>
#pragma GCC diagnostic pop
}


#define TRY(...)                                       \
	({                                                   \
		auto ret = __VA_ARGS__;                            \
		if(auto err = std::get_if<std::string>(&ret); err) \
			return std::move(*err);                          \
		std::move(std::get<0>(ret));                       \
	})

#define TRY_OPT(...)              \
	if(auto err = __VA_ARGS__; err) \
		return err;


namespace {
	std::variant<klapki::context::detail::bad_cow, std::string> unrelativise(const std::pair<std::string_view, std::string_view> & whom, const char * version,
	                                                                         const char * for_what) {
		if(whom.first[0] != '/') {
			// {version} {image|initrd} ...
			fmt::print(stderr, fgettext("{} {} has relative path {}/{}"), version, for_what, whom.first, whom.second);

			std::string reldir{whom.first};
			std::unique_ptr<char[], klapki::free_deleter> dir{realpath(reldir.c_str(), nullptr)};
			if(!dir) {
				fmt::print(stderr, "\n");
				// Getting realpath for {path}: {error}
				return fmt::format(fgettext("Getting realpath for {}: {}"), reldir, std::strerror(errno));
			}

			// continued from "has relative path"
			fmt::print(stderr, fgettext(", updating to {}/{}\n"), dir.get(), whom.second);
			return klapki::context::detail::bad_cow{std::string{dir.get()}};
		} else
			return klapki::context::detail::bad_cow{whom.first};
	}

	/// Actually just find lowest unoccupied
	std::variant<std::uint16_t, std::string> allocate_bootnum(const klapki::state::state & state) {
		std::uint16_t max{};
		if(std::begin(state.entries) == std::end(state.entries) || std::begin(state.entries)->first != 0)
			return static_cast<std::uint16_t>(0);
		else if(auto found = std::adjacent_find(std::begin(state.entries), std::end(state.entries),
		                                        [&](auto && prev, auto && next) {
			                                        max = next.first;
			                                        return next.first - prev.first != 1;
		                                        });
		        found != std::end(state.entries))
			return static_cast<std::uint16_t>(found->first + 1);
		else if(max != 0xFFFF)
			return static_cast<std::uint16_t>(max + 1);
		else
			return gettext("All boot entries occupied, can't allocate");
	}


	std::vector<std::pair<klapki::state::nonbase_dirname_t, klapki::state::shaa_t>>
	compact_initrd_dirs(std::string_view prev_dir, std::vector<klapki::context::detail::bad_cow> initrd_dirs) {
		std::vector<std::pair<klapki::state::nonbase_dirname_t, klapki::state::shaa_t>> ret;
		ret.reserve(initrd_dirs.size());

		for(auto && idir : initrd_dirs) {
			if(idir.get() == prev_dir)
				ret.emplace_back(klapki::state::nonbase_dirname_t{}, klapki::state::shaa_t{});
			else
				ret.emplace_back(idir.get(), klapki::state::shaa_t{});

			prev_dir = idir.get();
		}

		return ret;
	}
}


#define COPY_SHA(s) \
	{ s[0], s[1], s[2], s[3], s[4], s[5], s[6], s[7], s[8], s[9], s[10], s[11], s[12], s[13], s[14], s[15], s[16], s[17], s[18], s[19] }

std::optional<std::string> klapki::context::context::allocate_kernel_variant(const config & cfg, state::state & state, std::string version, std::string var,
                                                                             std::string kernel_dirname, sha_t & kernel_image_sha,
                                                                             std::vector<std::pair<state::nonbase_dirname_t, state::shaa_t>> initrd_dirnames,
                                                                             std::string image_basename,
                                                                             std::vector<std::pair<state::nonbase_dirname_t, std::string>> initrd_paths) {
	auto efi_base = fmt::format("{}\\{}\\{}\\", cfg.efi_root(), cfg.host, version);

	const auto new_bootnum = TRY(allocate_bootnum(state));
	if(cfg.verbose) {
		if(var.empty())
			fmt::print(fgettext("  Default variant assigned bootnum {:04X}\n"), new_bootnum);
		else
			fmt::print(fgettext("  Variant {} assigned bootnum {:04X}\n"), var, new_bootnum);
	}


	std::get<klapki::state::boot_order_structured>(state.order).ours.emplace_back(new_bootnum);
	state.statecfg.wanted_entries.emplace_back(state::stated_config_entry{
	    new_bootnum, {}, std::move(version), std::move(var), std::move(kernel_dirname), COPY_SHA(kernel_image_sha), std::move(initrd_dirnames)});

	state.entries.emplace(new_bootnum, state::boot_entry{{}, 0, {}, EFI_VARIABLE_NON_VOLATILE | EFI_VARIABLE_BOOTSERVICE_ACCESS | EFI_VARIABLE_RUNTIME_ACCESS});
	this->our_kernels.emplace(new_bootnum, our_kernel{"", "", {std::move(efi_base), std::move(image_basename)}, std::move(initrd_paths)});

	return {};
}

void klapki::context::context::purge_allocations_impl(state::state & state, const state::stated_config_entry & skern) {
	if(!state.entries.erase(skern.bootnum_hint))
		fmt::print(stderr, fgettext("Deleted entry {:04X} unknown in boot entries?\n"), skern.bootnum_hint);

	auto kern_n = this->our_kernels.extract(skern.bootnum_hint);
	if(!kern_n)
		throw __func__;
	auto && kern = kern_n.mapped();

	std::string_view dprev = this->deleted_files.emplace(std::move(kern.image_path)).first->first;
	for(auto && [didir, dibase] : kern.initrd_paths)
		if(didir)
			dprev = this->deleted_files.emplace(std::move(*didir), std::move(dibase)).first->first;
		else
			this->deleted_files.emplace(dprev, std::move(dibase));

	auto & our_boot_order = std::get<klapki::state::boot_order_structured>(state.order).ours;
	if(auto itr = std::find(std::begin(our_boot_order), std::end(our_boot_order), skern.bootnum_hint); itr != std::end(our_boot_order))
		our_boot_order.erase(itr);
	else
		fmt::print(stderr, fgettext("Deleted entry {:04X} unknown in boot order?\n"), skern.bootnum_hint);
}


std::optional<std::string> klapki::context::context::age(const config & cfg, state::state & state) {
	std::vector<fresh_kernel> fkerns;
	fkerns.swap(this->fresh_kernels);
	for(auto && fkern : std::move(fkerns)) {
		if(cfg.verbose)
			// version
			fmt::print(fgettext("Aging fresh kernel {}\n"), fkern.version);

		// only for "{} has relative path" above
		const auto image_dir = TRY(unrelativise(fkern.image, fkern.version.data(), gettext("image")));

		std::vector<detail::bad_cow> initrd_dirs_raw;
		initrd_dirs_raw.reserve(fkern.initrds.size());
		for(auto && initrd : fkern.initrds)
			// only for "{} has relative path" above
			initrd_dirs_raw.emplace_back(TRY(unrelativise(initrd, fkern.version.data(), gettext("initrd"))));
		const auto initrd_dirs = compact_initrd_dirs(image_dir.get(), std::move(initrd_dirs_raw));

		std::vector<std::pair<state::nonbase_dirname_t, std::string>> our_initrd_paths;
		our_initrd_paths.reserve(fkern.initrds.size());
		for(auto && initrd : fkern.initrds)
			our_initrd_paths.emplace_back(std::nullopt, initrd.second);

		sha_t nilsha{};
		TRY_OPT(this->allocate_kernel_variant(cfg, state, std::string{fkern.version}, "", std::string{image_dir.get()}, nilsha, initrd_dirs,
		                                      std::string{fkern.image.second}, our_initrd_paths));
		for(auto && var : state.statecfg.variants)
			TRY_OPT(this->allocate_kernel_variant(cfg, state, std::string{fkern.version}, var, std::string{image_dir.get()}, nilsha, initrd_dirs,
			                                      std::string{fkern.image.second}, our_initrd_paths));
	}

	return {};
}
