// SPDX-License-Identifier: MIT
/*! klapki::context::context::save(): step 9
 * Synthesise Boot1234 variable blobs from our entries in context::context and state::state
 * and flatten the BootOrder.
 *
 * Create a HD(...) path node for the ESP.
 *
 * For the join of context::context::our_kernels with state::state::statecfg::wanted_entries and state::state::entries:
 *   * construct, normalise, and toupper()ify the path under the ESP context::context::our_kernels::image_path -> "\KLAPKI\HOST\VERSION\VMLINUZ"
 *   * put that in a File(...) node and add it to the HD(...) node
 *   * construct the cmdline from context::context::our_kernels::initrd_paths ("initrd=" ::first "\" ::second " ")
 *                            and context::context::our_kernels::cmdline
 *     -> UTF-8->UCS-2
 * -> construct this all into state::state::entries::load_option and ::load_option_len
 * -> hash that into          state::state::entries::load_option_sha                   (detect changes in this step)
 *
 * Convert state::state::order from boot_order_structured to boot_order_flat:
 *   * sort our entries based on (state::state::statecfg::wanted_entries::version ↘,
 *                                state::state::statecfg::wanted_entries::variant's index in state::state::statecfg::variants ↗),
 *                               with the default ("") variant sorting before non-default variants
 *   * pick our insertion point based on max(state::state::statecfg::wanted_entries::boot_position, total amount of boot entries)
 *   * copy out the bit before our entries in the order, our entries in the order, the bit after our entries in the order
 */


#include "config.hpp"
#include "context.hpp"
#include "context_detail.hpp"
#include "iconv.hpp"
#include "util.hpp"
#include <algorithm>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
extern "C" {
#include <efiboot.h>
}

using namespace std::literals;


#define TRY_OPT(...)              \
	if(auto err = __VA_ARGS__; err) \
		return err;


namespace {
	// efidp_format_device_path takes a char * on bookworm and an unsigned char * on sid!
	template <class>
	struct first_arg;

	template <class F, class A, class... Rest>
	struct first_arg<F(A, Rest...)> {
		using first = A;
	};
}


std::string klapki::context::detail::fmt_devpath(const efidp_data * dp, ssize_t dp_len) {
	const auto size = efidp_format_device_path(nullptr, 0, dp, dp_len);
	if(size < 0)
		return __func__;
	else {
		std::string path(size - 1, '\0');  // size includes NUL
		efidp_format_device_path(reinterpret_cast<first_arg<decltype(efidp_format_device_path)>::first>(path.data()), size, dp, dp_len);
		return path;
	}
}


std::optional<std::string> klapki::context::context::save(const config & cfg, state::state & state, efidp_data * _test_esp_devpath_override) {
	std::vector<std::uint8_t> esp_devpath_raw;
	efidp_data * esp_devpath = _test_esp_devpath_override;
	if(!this->our_kernels.empty() && !esp_devpath) {
		do {
			esp_devpath_raw.resize(esp_devpath_raw.size() + 128);

			// extern ssize_t efi_generate_file_device_path(uint8_t *buf, ssize_t size,
			// 	      const char * const filepath,
			// 	      uint32_t options, ...)
			// EFIBOOT_ABBREV_HD matches what's produced by bootctl(1) install, and produces just HD()/File(),
			// however, this funxion requires the File() to exist, so by passing just the ESP root, we can append our potentially-not-yet-existent paths later on.
			if(auto size = efi_generate_file_device_path(esp_devpath_raw.data(), esp_devpath_raw.size(),  //
			                                             fmt::format("{}/", cfg.esp).c_str(),             //
			                                             EFIBOOT_ABBREV_HD);
			   size >= 0)
				esp_devpath_raw.resize(size);
			else if(errno != ENOSPC)
				return fmt::format("Making device path for ESP: {}", std::strerror(errno));
		} while(errno == ENOSPC);


		esp_devpath = reinterpret_cast<efidp_data *>(esp_devpath_raw.data());
		{  // esp_devpath is currently HD(some path)/File("\"). Trim it to just HD() for appending later
			efidp_data * fnode{};
			if(efidp_next_node(esp_devpath, const_cast<const efidp_data **>(&fnode)) != 1)
				throw __func__;

			fnode->type    = EFIDP_END_TYPE;
			fnode->subtype = EFIDP_END_ENTIRE;
		}


		if(cfg.verbose)
			// device path ("HD(...)")
			fmt::print(fgettext("ESP devpath: {}"), detail::fmt_devpath(reinterpret_cast<const efidp_data *>(esp_devpath_raw.data()), esp_devpath_raw.size()));
	}


	for(auto && [bootnum, kern] : this->our_kernels) {
		auto skern = std::find_if(std::begin(state.statecfg.wanted_entries), std::end(state.statecfg.wanted_entries),
		                          [&](auto && skern) { return skern.bootnum_hint == bootnum; });
		if(skern == std::end(state.statecfg.wanted_entries))
			throw __func__;
		auto bent = state.entries.find(bootnum);
		if(bent == std::end(state.entries))
			throw __func__;

		auto image_path = fmt::format("\\{}\\{}", kern.image_path.first, kern.image_path.second);
		std::replace(std::begin(image_path), std::end(image_path), '/', '\\');
		image_path.erase(std::unique(std::begin(image_path), std::end(image_path), [](auto l, auto r) { return l == '\\' && r == '\\'; }), std::end(image_path));
		std::transform(std::begin(image_path), std::end(image_path), std::begin(image_path), [](char c) { return std::toupper(c); });

		std::vector<std::uint8_t> devpath_file_node(efidp_make_file(nullptr, 0, image_path.data()));
		if(efidp_make_file(devpath_file_node.data(), devpath_file_node.size(), image_path.data()) < 0)
			return fmt::format("Entry {:04X}: creating devpath File(): {}", bootnum, std::strerror(errno));

		std::unique_ptr<efidp_data, klapki::free_deleter> devpath{({
			efidp_data * devpath_p;
			if(efidp_append_node(esp_devpath, reinterpret_cast<const efidp_data *>(devpath_file_node.data()), &devpath_p) < 0)
				return fmt::format("Entry {:04X}: appending File(): {}", bootnum, std::strerror(errno));
			devpath_p;
		})};
		const auto devpath_len = efidp_size(devpath.get());


		if(cfg.verbose)
			// device path ("HD(...)\File(...)")
			fmt::print(fgettext("Entry {:04X} devpath: {}\n"), bootnum, detail::fmt_devpath(devpath.get(), devpath_len));

		std::vector<char> cmdline;
		auto append = [&](const std::string_view & bit) {
			iconv<"UTF-8", "UCS-2", char>(bit.data(), bit.size(),
			                              [&](auto && segment) { cmdline.insert(std::end(cmdline), std::begin(segment), std::end(segment)); });
			if(cfg.verbose)
				fmt::print("{}", bit);
		};
		if(cfg.verbose)
			// cmdline (root=asd ...) follows piece-meal
			fmt::print(fgettext("Entry {:04X} cmdline: "), bootnum);
		// Must be at start, we use position in derive() to match extraneous ones from cmdline
		std::string_view prev = kern.image_path.first;
		for(auto && ipath : kern.initrd_paths) {
			if(ipath.first)
				prev = *ipath.first;
			append("initrd="sv);
			append(prev);
			append((prev.back() == '\\' || ipath.second.front() == '\\') ? ""sv : "\\"sv);
			append(ipath.second);
			append(" "sv);
		}
		append(kern.cmdline);
		if(cfg.verbose)
			fmt::print("\n");


		// extern ssize_t efi_loadopt_create(uint8_t *buf, ssize_t size,
		//				  uint32_t attributes, efidp dp,
		//				  ssize_t dp_size, unsigned char *description,
		//				  uint8_t *optional_data,
		//				  size_t optional_data_size)
		bent->second.load_option_len = efi_loadopt_create(nullptr, 0,                                                  //
		                                                  bent->second.attributes,                                     //
		                                                  devpath.get(), devpath_len,                                  //
		                                                  reinterpret_cast<unsigned char *>(kern.description.data()),  //
		                                                  reinterpret_cast<std::uint8_t *>(cmdline.data()), cmdline.size());

		bent->second.load_option = std::make_shared_for_overwrite<std::uint8_t[]>(bent->second.load_option_len);
		if(efi_loadopt_create(bent->second.load_option.get(), bent->second.load_option_len,  //
		                      bent->second.attributes,                                       //
		                      devpath.get(), devpath_len,                                    //
		                      reinterpret_cast<unsigned char *>(kern.description.data()),    //
		                      reinterpret_cast<std::uint8_t *>(cmdline.data()), cmdline.size()) < 0)
			return fmt::format("Making load option for {:04X}: {}", bootnum, std::strerror(errno));

		klapki::SHA1(bent->second.load_option.get(), bent->second.load_option_len, bent->second.load_option_sha);
		if(std::memcmp(bent->second.load_option_sha, skern->load_option_sha, sizeof(sha_t)))
			fmt::print(fgettext("Entry {:04X} changed\n"), bootnum);
		std::memcpy(skern->load_option_sha, bent->second.load_option_sha, sizeof(sha_t));
	}


	if(cfg.verbose)
		// Written just before "Bootorder post" below, the format specifiers should align horizontally
		fmt::print(fgettext("Bootorder pre : {}\n"), state.order);
	state.order = std::visit(
	    klapki::overload{
	        [&](klapki::state::boot_order_flat && bof) {
		        fmt::print(stderr, "wisen(): flat bootorder?\n");  // Weird, but that's what we want anyway
		        return std::move(bof);
	        },
	        [&](klapki::state::boot_order_structured && bos) {
		        // By biggest version, then variant index.
		        std::sort(std::begin(bos.ours), std::end(bos.ours), [&](auto && lhs, auto && rhs) {
			        auto lskern = std::find_if(std::begin(state.statecfg.wanted_entries), std::end(state.statecfg.wanted_entries),
			                                   [&](auto && skern) { return skern.bootnum_hint == lhs; });
			        if(lskern == std::end(state.statecfg.wanted_entries))
				        throw __func__;
			        auto rskern = std::find_if(std::begin(state.statecfg.wanted_entries), std::end(state.statecfg.wanted_entries),
			                                   [&](auto && skern) { return skern.bootnum_hint == rhs; });
			        if(rskern == std::end(state.statecfg.wanted_entries))
				        throw __func__;

			        if(auto cmp = strverscmp(lskern->version.c_str(), rskern->version.c_str()); cmp)
				        return cmp > 0;

			        auto lvar = lskern->variant == "" ? -1 : std::find_if(std::begin(state.statecfg.variants), std::end(state.statecfg.variants), [&](auto && var) {
				                                                 return var == lskern->variant;
			                                                 }) - std::begin(state.statecfg.variants);
			        auto rvar = rskern->variant == "" ? -1 : std::find_if(std::begin(state.statecfg.variants), std::end(state.statecfg.variants), [&](auto && var) {
				                                                 return var == rskern->variant;
			                                                 }) - std::begin(state.statecfg.variants);

			        return lvar < rvar;
		        });


		        const auto target_pos = std::min(bos.foreign.size(), static_cast<std::size_t>(state.statecfg.boot_position));
		        if(bos.foreign.size() < state.statecfg.boot_position)
			        // If the total count of boot entries is smaller than the configured boot position.
			        fmt::print(stderr, fgettext("Not enough entries to be at position {}. Being at {} instead.\n"), state.statecfg.boot_position, target_pos);

		        const auto size = bos.foreign.size() + bos.ours.size();
		        auto flat       = std::make_shared_for_overwrite<std::uint16_t[]>(size);

		        auto curs = std::copy_n(bos.foreign.data(), target_pos, flat.get());
		        curs      = std::copy_n(bos.ours.data(), bos.ours.size(), curs);
		        curs      = std::copy_n(bos.foreign.data() + target_pos, bos.foreign.size() - target_pos, curs);
		        if(curs != flat.get() + size)  // This is an assert() but asserts blow ass, so it's a throw instead
			        throw __func__;

		        return state::boot_order_flat{flat, size};
	        },
	    },
	    std::move(state.order));
	if(cfg.verbose)
		fmt::print(fgettext("Bootorder post: {}\n"), state.order);

	return {};
}
