// SPDX-License-Identifier: MIT
/*! klapki::context::context::wisen(): step 8
 * Execute /etc/klapki (or whatever config::wisdom_root()) to fill out data in context::context::our_kernels::{description,cmdline},
 * based on the output state::state::statecfg::wanted_entries' ::version and ::variant.
 */


#include "config.hpp"
#include "context.hpp"
#include "context_detail.hpp"
#include "vore-file"
#include "vore-token"
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

using namespace std::literals;


#define TRY(...)                                       \
	({                                                   \
		auto ret = __VA_ARGS__;                            \
		if(auto err = std::get_if<std::string>(&ret); err) \
			return std::move(*err);                          \
		std::move(std::get<0>(ret));                       \
	})


namespace {
	struct launched_wisdom {
		pid_t pid;
		vore::file::fd rdpipe;
		const char * from;
		std::string & into;
	};
	struct wisdom {
		std::string data;
	};

	std::variant<launched_wisdom, std::string> launch_wisdom(const klapki::config & cfg, const char * from, const char * version, const char * variant,
	                                                         std::string & into) {
		int comm[2];
		if(pipe2(comm, O_CLOEXEC) == -1)
			return fmt::format("opening {} {} {} pipe: {}", from, version, variant, std::strerror(errno));

		const auto hook = fmt::format("{}{}", cfg.wisdom_root(), from);
		switch(auto pid = vfork()) {
			case -1:
				close(comm[0]);
				close(comm[1]);
				return fmt::format("forking for {} {} {}: {}", from, version, variant, std::strerror(errno));

			case 0: {  // child
				dup2(comm[1], 1);

				execl(hook.c_str(), from, version, variant, (char *)nullptr);
				auto err = errno;
				std::fprintf(stderr, "exec %.*s %s %s: %s\n", (int)hook.size(), hook.data(), version, variant, std::strerror(err));
				_exit(err == ENOENT ? 127 : 126);
			}

			default: {  // parent
				close(comm[1]);
				return launched_wisdom{pid, comm[0], from, into};
			}
		}
	}

	std::optional<std::string> collect_wisdom(launched_wisdom launched) {
		auto && [pid, rdpipe, from, into] = launched;
		into.clear();

		char buf[64 * 1024];
		for(ssize_t rd = -1; rd;) {
			while((rd = read(rdpipe, buf, sizeof(buf))) == -1 && errno == EINTR)
				;
			if(rd == -1)
				return fmt::format("{} read failed with {}", from, std::strerror(errno));

			into.append(buf, rd);
		}

		int result;
		waitpid(pid, &result, 0);
		if(WIFEXITED(result) && WEXITSTATUS(result) != 0)
			// {description|cmdline} ... {1}
			return fmt::format(fgettext("{} child returned {}"), gettext(from), WEXITSTATUS(result));
		else if(WIFSIGNALED(result))
			return fmt::format(fgettext("{} child killed by signal {} ({})"), gettext(from), WTERMSIG(result), strsignal(WTERMSIG(result)));

		if(auto idx = into.find('\0'); idx != std::string::npos)  // end at first NUL, if any; prevent unexpected behaviour by behaving like a C string
			into.erase(idx);

		// Character set stolen from isspace()
		into.erase(into.find_last_not_of(" \f\n\r\t\v") + 1);
		into.erase(0, into.find_first_not_of(" \f\n\r\t\v"));

		std::replace(std::begin(into), std::end(into), '\n', ' ');

		return {};
	}

	void validate_cmdline(const std::string & cmdline) {
		for(auto && tok : vore::soft_tokenise{cmdline, " \f\n\r\t\v"sv})
			if(tok.starts_with("initrd="sv)) {
				// "initrd=\abc\def\..."
				fmt::print(stderr, fgettext("Stray {} in cmdline, things might not work as planned\n"), tok);
				break;
			}
	}
}

#define gettext_noop(str) str
std::optional<std::string> klapki::context::context::wisen(const config & cfg, const state::state & state) {
	std::vector<launched_wisdom> launched;
	launched.reserve(this->our_kernels.size() * 2);

	for(auto && [bootnum, kern] : this->our_kernels) {
		auto skern = std::find_if(std::begin(state.statecfg.wanted_entries), std::end(state.statecfg.wanted_entries),
		                          [&](auto && skern) { return skern.bootnum_hint == bootnum; });
		if(skern == std::end(state.statecfg.wanted_entries))
			throw __func__;

		// both only for "child returned/killed" above
		launched.emplace_back(TRY(launch_wisdom(cfg, gettext_noop("description"), skern->version.c_str(), skern->variant.c_str(), kern.description)));
		launched.emplace_back(TRY(launch_wisdom(cfg, gettext_noop("cmdline"), skern->version.c_str(), skern->variant.c_str(), kern.cmdline)));
	}

	for(auto && l : launched)
		collect_wisdom(std::move(l));

	for(auto && [_, kern] : this->our_kernels)
		validate_cmdline(kern.cmdline);

	return {};
}
