// SPDX-License-Identifier: MIT


#include "config.hpp"
#include "context.hpp"
#include "state.hpp"
#include "util.hpp"
#include <clocale>
#include <fmt/format.h>


#define TRY(ec, ...)                                           \
	({                                                           \
		auto ret = __VA_ARGS__;                                    \
		if(auto errmsg = std::get_if<std::string>(&ret); errmsg) { \
			fmt::print(stderr, "{}\n", *errmsg);                     \
			return ec;                                               \
		}                                                          \
		std::move(std::get<0>(ret));                               \
	})

#define TRY_OPT(ec, pref, ...)          \
	if(auto err = __VA_ARGS__; err) {     \
		fmt::print("{}: {}\n", pref, *err); \
		return ec;                          \
	}


namespace klapki {
	static int main(int argc, char * const * argv) {
		std::setlocale(LC_ALL, "");
		bindtextdomain("klapki", "/usr/share/locale");
		// bindtextdomain("klapki", "out/locale");
		textdomain("klapki");


		auto cfg = TRY(1, config::read(argc, argv));
		if(cfg.verbose)
			fmt::print("{}\n", cfg);


		const auto input_state = TRY(2, state::state::load(cfg.host, cfg.compress));


		auto output_state = TRY(3, context::resolve_state_context(cfg, input_state));


		auto context = context::context::derive(cfg, output_state);


		for(auto && op : cfg.ops) {
			if(cfg.verbose)
				// {op}
				fmt::print(fgettext("Running {}\n"), klapki::op{op});

			TRY_OPT(4, klapki::op{op}, op::execute(op, cfg, output_state, context));
		}


		TRY_OPT(5, "context.propagate()", context.propagate(cfg, output_state));
		if(cfg.verbose) {
			fmt::print(fgettext("Propagated:"));
			op::execute(klapki::ops::dump{}, cfg, output_state, context);
		}

		TRY_OPT(6, "context.age()", context.age(cfg, output_state));
		if(cfg.verbose) {
			fmt::print(fgettext("Aged:"));
			op::execute(klapki::ops::dump{}, cfg, output_state, context);
		}

		TRY_OPT(7, "context.wisen()", context.wisen(cfg, output_state));
		if(cfg.verbose) {
			fmt::print(fgettext("Wisened:"));
			op::execute(klapki::ops::dump{}, cfg, output_state, context);
		}

		TRY_OPT(8, "context.save()", context.save(cfg, output_state));
		if(cfg.verbose) {
			fmt::print(fgettext("Saved:"));
			op::execute(klapki::ops::dump{}, cfg, output_state, context);
		}

		if(cfg.commit) {  // Both const, context only updates SHAs which we don't print anyway
			TRY_OPT(9, "context.commit()", context.commit(cfg, output_state));
			TRY_OPT(10, "output_state.commit()", output_state.commit(cfg.host.c_str(), cfg.compress, input_state));
		}

		return 0;
	}
}

int main(int argc, char * const * argv) {
	return klapki::main(argc, argv);
}
