// SPDX-License-Identifier: MIT
/*! klapki::op::execute(): step 5
 * Alter the output klapki::state::state and klapki::context::context:
 *   klapki::ops::dump:       just dump
 *   klapki::ops::bootpos:    edit state::statecfg::boot_position
 *   klapki::ops::delkernel:  remove context::fresh_kernels with matching ::version and purge_allocations() with matching ::version
 *                            (remove cached Boot1234 entries from state::entries,
 *                             remove from known kernels in state::our_kernels, but
 *                             save the files to delete from the ESP (image, initrds) in context::deleted_files,
 *                             delete the entry from BootOrder state::order)
 *   klapki::ops::addkernel:  append to context::fresh_kernels unless state::statecfg::wanted_entries or context::fresh_kernels already contains this version,
 *                            splitting kernel and initrd paths as ("/a/b/c", "vmli...") or (".", "vmli...")
 *   klapki::ops::addvariant: append to state::statecfg::variants unless empty (default) or already in there
 *   klapki::ops::delvariant: remove from state::statecfg::variants
 */


#include "config.hpp"
#include "context.hpp"
#include "ops.hpp"
#include <fmt/format.h>

using namespace std::literals;


namespace {
	std::pair<std::string_view, std::string_view> slash_path(const std::string_view & path) {
		if(auto last_slash = path.rfind('/'); last_slash != std::string::npos)
			return {path.substr(0, last_slash ? last_slash : 1),  // drop trailing slash here
			        path.substr(last_slash + 1)};
		else
			return {"."sv,  // age() will resolve this to abspath
			        path};
	}
}


std::optional<std::string> klapki::ops::execute(const klapki::ops::dump &, const klapki::config &, klapki::state::state & state,
                                                klapki::context::context & context) {
	fmt::print(fgettext("\n"
	                    "Boot order: {}\n"
	                    "{} boot entries\n"
	                    "Desired boot position: {}\n"
	                    "Boot variants: "),
	           state.order, state.entries.size(), state.statecfg.boot_position);

	if(state.statecfg.variants.empty())
		// none variants; otherwise printed piece-meal
		fmt::print(fgettext("(none)"));
	else {
		bool first = true;
		for(auto && el : state.statecfg.variants) {
			if(!first)
				fmt::print(", ");
			else
				first = false;

			fmt::print("{}", el);
		}
	}
	fmt::print(fgettext("\n"
	                    "Wanted entries: ["));

	bool first = true;
	for(auto && el : state.statecfg.wanted_entries) {
		if(first) {
			fmt::print("\n");
			first = false;
		}

		fmt::print("  {}\n", el);
	}

	fmt::print("]\n"
	           "\n"
	           "{}\n"
	           "\n",
	           context);

	return {};
}

std::optional<std::string> klapki::ops::execute(const klapki::ops::bootpos & bp, const klapki::config &, klapki::state::state & state,
                                                klapki::context::context &) {
	state.statecfg.boot_position = bp.position;
	return {};
}

std::optional<std::string> klapki::ops::execute(const klapki::ops::addkernel & ak, const klapki::config &, klapki::state::state & state,
                                                klapki::context::context & context) {
	if(std::any_of(std::begin(state.statecfg.wanted_entries), std::end(state.statecfg.wanted_entries),
	               [&](auto && went) { return went.version == ak.version; }) ||
	   std::any_of(std::begin(context.fresh_kernels), std::end(context.fresh_kernels), [&](auto && fk) { return fk.version == ak.version; })) {
		fmt::print(stderr, fgettext("addkernel: kernel version {} already known\n"), ak.version);
		return {};
	}

	klapki::context::fresh_kernel kern{ak.version, slash_path(ak.image), {}};

	kern.initrds.reserve(ak.initrds.size());
	std::transform(std::begin(ak.initrds), std::end(ak.initrds), std::back_inserter(kern.initrds), slash_path);

	context.fresh_kernels.emplace_back(std::move(kern));
	return {};
}

std::optional<std::string> klapki::ops::execute(const klapki::ops::delkernel & dk, const klapki::config &, klapki::state::state & state,
                                                klapki::context::context & context) {
	context.purge_allocations(state, [&](auto && skern) { return skern.version == dk.version; });

	context.fresh_kernels.erase(
	    std::remove_if(std::begin(context.fresh_kernels), std::end(context.fresh_kernels), [&](auto && fkern) { return fkern.version == dk.version; }),
	    std::end(context.fresh_kernels));

	return {};
}

std::optional<std::string> klapki::ops::execute(const klapki::ops::addvariant & av, const klapki::config &, klapki::state::state & state,
                                                klapki::context::context &) {
	if(av.variant[0] == '\0' ||
	   std::find(std::begin(state.statecfg.variants), std::end(state.statecfg.variants), av.variant) != std::end(state.statecfg.variants))
		fmt::print(stderr, fgettext("addvariant: boot variant {} already known\n"), av.variant[0] == '\0' ? "(default)" : av.variant);
	else
		state.statecfg.variants.push_back(av.variant);

	return {};
}

std::optional<std::string> klapki::ops::execute(const klapki::ops::delvariant & dv, const klapki::config &, klapki::state::state & state,
                                                klapki::context::context &) {
	if(auto itr = std::find(std::begin(state.statecfg.variants), std::end(state.statecfg.variants), dv.variant); itr != std::end(state.statecfg.variants))
		state.statecfg.variants.erase(itr);
	else
		fmt::print(stderr, fgettext("delvariant: boot variant {} not known\n"), dv.variant);

	return {};
}
