// SPDX-License-Identifier: MIT
/*! klapki::state::state::load(): step 2
 * Read from /sys/firmware/efi/efivars/:
 *   Boot1234                 -> state::entries (just the raw unhashed data)
 *   BootOrder                -> state::order (boot_order_flat)
 *   {host} {efi_guid_klapki} -> state::stated_config::parse() -> state::statecfg
 */


#include "state.hpp"
#include "efi.hpp"
#include <cstdlib>
#include <fmt/format.h>
#include <memory>
#include <optional>
#include <utility>
extern "C" {
#pragma GCC diagnostic push
#if __clang__
#pragma GCC diagnostic ignored "-Wgnu-zero-variadic-macro-arguments"
#endif
#pragma GCC diagnostic ignored "-Wvariadic-macros"
#include <efivar.h>
#pragma GCC diagnostic pop
// Doesn't include efivar.h:
#include <efivar-guids.h>
}

using namespace std::literals;


namespace {
	bool operator==(const efi_guid_t & lhs, const efi_guid_t & rhs) noexcept {
		return !std::memcmp(&lhs, &rhs, sizeof(efi_guid_t));
	}

	bool is_boot_entry(const efi_guid_t & guid, const std::string_view & name) noexcept {
		return guid == efi_guid_global && name.size() == 8 && name.starts_with("Boot"sv) &&  //
		       std::isxdigit(name[4]) && std::isxdigit(name[5]) && std::isxdigit(name[6]) && std::isxdigit(name[7]);
	}

	bool is_boot_order(const efi_guid_t & guid, const std::string_view & name) noexcept {
		return guid == efi_guid_global && name == "BootOrder"sv;
	}

	bool is_our_config(const efi_guid_t & guid, const std::string_view & name, std::string_view us) noexcept {
		return guid == klapki::efi_guid_klapki && name == us;
	}


	std::optional<std::tuple<std::shared_ptr<std::uint8_t[]>, std::size_t, std::uint32_t>> get_efi_data(const efi_guid_t & guid, const char * name) {
		std::uint8_t * raw_data{};
		std::size_t size{};
		std::uint32_t attr{};
		if(efi_get_variable(guid, name, &raw_data, &size, &attr) < 0)
			return std::nullopt;

		return std::tuple{std::shared_ptr<std::uint8_t[]>{raw_data, std::free}, size, attr};
	}

	bool get_boot_order(klapki::state::boot_order_flat & bord) {
		auto dt = get_efi_data(efi_guid_global, "BootOrder");
		if(!dt)
			return false;
		auto && [data, size, _] = *dt;

		bord.order     = std::reinterpret_pointer_cast<std::uint16_t[]>(data);
		bord.order_cnt = size / 2;
		return true;
	}

	bool get_boot_entry(std::map<std::uint16_t, klapki::state::boot_entry> & bents, const char * name) {
		auto dt = get_efi_data(efi_guid_global, name);
		if(!dt)
			return false;
		auto && [data, size, attr] = *dt;

		bents.emplace(std::strtoul(name + "Boot"sv.size(), nullptr, 16), klapki::state::boot_entry{std::move(data), size, {}, attr});
		return true;
	}

	bool get_our_config(klapki::state::stated_config & statecfg, const char * us, bool may_decompress_statecfg) {
		auto dt = get_efi_data(klapki::efi_guid_klapki, us);
		if(!dt)
			return false;
		auto && [data, size, _] = *dt;

		klapki::state::stated_config::parse(statecfg, may_decompress_statecfg, data.get(), size);
		return true;
	}
}


std::variant<klapki::state::state, std::string> klapki::state::state::load(const std::string_view & us, bool may_decompress_statecfg) {
	if(!efi_variables_supported())
		return gettext("EFI not supported?");


	klapki::state::state ret{};
	bool saw_boot_order{}, saw_our_config{};
	{
		efi_guid_t * guid{};
		char * name_i{};
		for(int err; (err = efi_get_next_variable_name(&guid, &name_i));)
			if(err < 0)
				return fmt::format("EFI load: iteration: {}", std::strerror(errno));
			else {
				std::string_view name{name_i};
				bool err{};

				if(is_boot_entry(*guid, name))
					err = !get_boot_entry(ret.entries, name.data());
				else if(is_boot_order(*guid, name) && !std::exchange(saw_boot_order, true))
					err = !get_boot_order(ret.order.emplace<boot_order_flat>());
				else if(is_our_config(*guid, name, us) && !std::exchange(saw_our_config, true))
					err = !get_our_config(ret.statecfg, us.data(), may_decompress_statecfg);

				if(err)
					return fmt::format("EFI load: getting {}: {}", name, std::strerror(errno));
			}
	}

	if(!saw_boot_order)
		fmt::print(stderr, fgettext("EFI: no BootOrder?\n"));
	if(!saw_our_config)
		fmt::print(stderr, fgettext("EFI: no config for this host ({}) found; going to the top\n"), us);
	return ret;
}


bool klapki::state::stated_config_entry::operator==(const klapki::state::stated_config_entry & other) const noexcept {
	return this->bootnum_hint == other.bootnum_hint &&                                     //
	       !std::memcmp(this->load_option_sha, other.load_option_sha, sizeof(sha_t)) &&    //
	       this->version == other.version &&                                               //
	       this->variant == other.variant &&                                               //
	       this->kernel_dirname == other.kernel_dirname &&                                 //
	       !std::memcmp(this->kernel_image_sha, other.kernel_image_sha, sizeof(sha_t)) &&  //
	       this->initrd_dirnames == other.initrd_dirnames;
}

bool klapki::state::stated_config::operator==(const klapki::state::stated_config & other) const noexcept {
	return this->boot_position == other.boot_position && this->variants == other.variants && this->wanted_entries == other.wanted_entries;
}
