// SPDX-License-Identifier: MIT


#define NAME "sr"
#include "everything.hpp"


// # efibootmgr -v
// BootCurrent: 0002
// Timeout: 0 seconds
// BootOrder: 0009,0000,0001,0002,0003,0004,0005,0006,0007,0008
// Boot0000* UiApp FvVol(7cb8bdc9-f8eb-4f34-aaea-3ee4af6516a1)/FvFile(462caa21-7614-4503-836e-8ab6f4662331)
// Boot0001* UEFI QEMU DVD-ROM QM00003     PciRoot(0x0)/Pci(0x1,0x1)/Ata(1,0,0)N.....YM....R,Y.
// Boot0002* UEFI Misc Device      PciRoot(0x0)/Pci(0x4,0x0)N.....YM....R,Y.
// Boot0003* UEFI PXEv4 (MAC:525400123456) PciRoot(0x0)/Pci(0x3,0x0)/MAC(525400123456,1)N.....YM....R,Y.
// Boot0004* UEFI PXEv4 (MAC:525400123456) 2       PciRoot(0x0)/Pci(0x3,0x0)/MAC(525400123456,1)/IPv4(0.0.0.00.0.0.0,0,0)N.....YM....R,Y.
// Boot0005* UEFI PXEv6 (MAC:525400123456) PciRoot(0x0)/Pci(0x3,0x0)/MAC(525400123456,1)/IPv6([::]:<->[::]:,0,0)N.....YM....R,Y.
// Boot0006* UEFI HTTPv4 (MAC:525400123456)        PciRoot(0x0)/Pci(0x3,0x0)/MAC(525400123456,1)/IPv4(0.0.0.00.0.0.0,0,0)/Uri()N.....YM....R,Y.
// Boot0007* UEFI HTTPv6 (MAC:525400123456)        PciRoot(0x0)/Pci(0x3,0x0)/MAC(525400123456,1)/IPv6([::]:<->[::]:,0,0)/Uri()N.....YM....R,Y.
// Boot0008* EFI Internal Shell    FvVol(7cb8bdc9-f8eb-4f34-aaea-3ee4af6516a1)/FvFile(7c04a583-9e3e-4f1c-ad65-e05268d0b4d1)
// Boot0009* debian        HD(1,GPT,6bd84995-b9a4-47a5-ab45-4dfffa6be35b,0x800,0x2f000)/File(\EFI\debian\shimx64.efi)
//
// # for f in Boot000*; do { head -c 4 > /dev/null; cat; } < $f | sha1sum; echo $f; done | paste - -
// 22a4f6ee9af6dba01d3528deb64b74b582fc182b  -     Boot0000-8be4df61-93ca-11d2-aa0d-00e098032b8c
// 0b5d2c98ac5de6148a4a1490ff9d5df69039f04e  -     Boot0001-8be4df61-93ca-11d2-aa0d-00e098032b8c
// 8f70c56c75d06c777146859573c2aa343700c552  -     Boot0002-8be4df61-93ca-11d2-aa0d-00e098032b8c
// 20bd5f402271d57a88ea314fe35c1705956b1f74  -     Boot0003-8be4df61-93ca-11d2-aa0d-00e098032b8c
// f34fffcd65e61d17c75097571afbcdccd43dd7f4  -     Boot0004-8be4df61-93ca-11d2-aa0d-00e098032b8c
// effe799f8a02beea20a1531de04ec5d9626b5b51  -     Boot0005-8be4df61-93ca-11d2-aa0d-00e098032b8c
// b1e137e887d806f10ca7b6d7da833cd987186e5e  -     Boot0006-8be4df61-93ca-11d2-aa0d-00e098032b8c
// ec8c71c21184021fab57c32687145a1704cc028c  -     Boot0007-8be4df61-93ca-11d2-aa0d-00e098032b8c
// df5d6605cb8f4366d745a8464cfb26c1efdc305c  -     Boot0008-8be4df61-93ca-11d2-aa0d-00e098032b8c
// baf8522ef57743ec4ed5aa6921b3ff833d263ec9  -     Boot0009-8be4df61-93ca-11d2-aa0d-00e098032b8c


namespace {
	const klapki::config config{
	    .host            = "sr"s,
	    .verbose         = false,
	    .commit          = true,
	    .esp             = "/boot/efi"sv,
	    .ops             = {klapki::ops::addkernel{"version", "/vmlinuz", {"/boot/initrd", "/boot/initrd2", "initrd3"}}, klapki::ops::bootpos{4}},
	    .efi_root_raw    = ""sv,
	    .wisdom_root_raw = "/tmp/" NAME "/"sv,
	};
	const std::string_view config_wisdom_description = "echo \"$@\"";
	const std::string_view config_wisdom_cmdline     = "exec sed -E 's/initrd=[^ ]+ ?//g' /proc/cmdline";

	const klapki::state::state from_step_2{
	    klapki::state::boot_order_flat{
	        std::shared_ptr<std::uint16_t[]>{new std::uint16_t[]{0x0009, 0x0000, 0x0001, 0x0002, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007, 0x0008}}, 10},
	    {
	        // Boot0000-8be4df61-93ca-11d2-aa0d-00e098032b8c
	        {0x0000, boot_entry_raw({0x07, 0x00, 0x00, 0x00, 0x09, 0x01, 0x00, 0x00, 0x2c, 0x00, 0x55, 0x00, 0x69, 0x00, 0x41, 0x00, 0x70,
	                                 0x00, 0x70, 0x00, 0x00, 0x00, 0x04, 0x07, 0x14, 0x00, 0xc9, 0xbd, 0xb8, 0x7c, 0xeb, 0xf8, 0x34, 0x4f,
	                                 0xaa, 0xea, 0x3e, 0xe4, 0xaf, 0x65, 0x16, 0xa1, 0x04, 0x06, 0x14, 0x00, 0x21, 0xaa, 0x2c, 0x46, 0x14,
	                                 0x76, 0x03, 0x45, 0x83, 0x6e, 0x8a, 0xb6, 0xf4, 0x66, 0x23, 0x31, 0x7f, 0xff, 0x04, 0x00})},
	        // Boot0001-8be4df61-93ca-11d2-aa0d-00e098032b8c
	        {0x0001, boot_entry_raw({0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x55, 0x00, 0x45, 0x00, 0x46, 0x00, 0x49, 0x00, 0x20,
	                                 0x00, 0x51, 0x00, 0x45, 0x00, 0x4d, 0x00, 0x55, 0x00, 0x20, 0x00, 0x44, 0x00, 0x56, 0x00, 0x44, 0x00, 0x2d, 0x00,
	                                 0x52, 0x00, 0x4f, 0x00, 0x4d, 0x00, 0x20, 0x00, 0x51, 0x00, 0x4d, 0x00, 0x30, 0x00, 0x30, 0x00, 0x30, 0x00, 0x30,
	                                 0x00, 0x33, 0x00, 0x20, 0x00, 0x00, 0x00, 0x02, 0x01, 0x0c, 0x00, 0xd0, 0x41, 0x03, 0x0a, 0x00, 0x00, 0x00, 0x00,
	                                 0x01, 0x01, 0x06, 0x00, 0x01, 0x01, 0x03, 0x01, 0x08, 0x00, 0x01, 0x00, 0x00, 0x00, 0x7f, 0xff, 0x04, 0x00, 0x4e,
	                                 0xac, 0x08, 0x81, 0x11, 0x9f, 0x59, 0x4d, 0x85, 0x0e, 0xe2, 0x1a, 0x52, 0x2c, 0x59, 0xb2})},
	        // Boot0002-8be4df61-93ca-11d2-aa0d-00e098032b8c
	        {0x0002, boot_entry_raw({0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x16, 0x00, 0x55, 0x00, 0x45, 0x00, 0x46, 0x00, 0x49, 0x00, 0x20, 0x00, 0x4d,
	                                 0x00, 0x69, 0x00, 0x73, 0x00, 0x63, 0x00, 0x20, 0x00, 0x44, 0x00, 0x65, 0x00, 0x76, 0x00, 0x69, 0x00, 0x63, 0x00, 0x65, 0x00,
	                                 0x00, 0x00, 0x02, 0x01, 0x0c, 0x00, 0xd0, 0x41, 0x03, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x06, 0x00, 0x00, 0x04, 0x7f,
	                                 0xff, 0x04, 0x00, 0x4e, 0xac, 0x08, 0x81, 0x11, 0x9f, 0x59, 0x4d, 0x85, 0x0e, 0xe2, 0x1a, 0x52, 0x2c, 0x59, 0xb2})},
	        // Boot0003-8be4df61-93ca-11d2-aa0d-00e098032b8c
	        {0x0003, boot_entry_raw({0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x3b, 0x00, 0x55, 0x00, 0x45, 0x00, 0x46, 0x00, 0x49, 0x00, 0x20, 0x00, 0x50,
	                                 0x00, 0x58, 0x00, 0x45, 0x00, 0x76, 0x00, 0x34, 0x00, 0x20, 0x00, 0x28, 0x00, 0x4d, 0x00, 0x41, 0x00, 0x43, 0x00, 0x3a, 0x00,
	                                 0x35, 0x00, 0x32, 0x00, 0x35, 0x00, 0x34, 0x00, 0x30, 0x00, 0x30, 0x00, 0x31, 0x00, 0x32, 0x00, 0x33, 0x00, 0x34, 0x00, 0x35,
	                                 0x00, 0x36, 0x00, 0x29, 0x00, 0x00, 0x00, 0x02, 0x01, 0x0c, 0x00, 0xd0, 0x41, 0x03, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01,
	                                 0x06, 0x00, 0x00, 0x03, 0x03, 0x0b, 0x25, 0x00, 0x52, 0x54, 0x00, 0x12, 0x34, 0x56, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	                                 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x7f,
	                                 0xff, 0x04, 0x00, 0x4e, 0xac, 0x08, 0x81, 0x11, 0x9f, 0x59, 0x4d, 0x85, 0x0e, 0xe2, 0x1a, 0x52, 0x2c, 0x59, 0xb2})},
	        // Boot0004-8be4df61-93ca-11d2-aa0d-00e098032b8c
	        {0x0004, boot_entry_raw({0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x56, 0x00, 0x55, 0x00, 0x45, 0x00, 0x46, 0x00, 0x49, 0x00, 0x20, 0x00,
	                                 0x50, 0x00, 0x58, 0x00, 0x45, 0x00, 0x76, 0x00, 0x34, 0x00, 0x20, 0x00, 0x28, 0x00, 0x4d, 0x00, 0x41, 0x00, 0x43, 0x00,
	                                 0x3a, 0x00, 0x35, 0x00, 0x32, 0x00, 0x35, 0x00, 0x34, 0x00, 0x30, 0x00, 0x30, 0x00, 0x31, 0x00, 0x32, 0x00, 0x33, 0x00,
	                                 0x34, 0x00, 0x35, 0x00, 0x36, 0x00, 0x29, 0x00, 0x20, 0x00, 0x32, 0x00, 0x00, 0x00, 0x02, 0x01, 0x0c, 0x00, 0xd0, 0x41,
	                                 0x03, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x06, 0x00, 0x00, 0x03, 0x03, 0x0b, 0x25, 0x00, 0x52, 0x54, 0x00, 0x12,
	                                 0x34, 0x56, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	                                 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x03, 0x0c, 0x1b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	                                 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0x04, 0x00,
	                                 0x4e, 0xac, 0x08, 0x81, 0x11, 0x9f, 0x59, 0x4d, 0x85, 0x0e, 0xe2, 0x1a, 0x52, 0x2c, 0x59, 0xb2})},
	        // Boot0005-8be4df61-93ca-11d2-aa0d-00e098032b8c
	        {0x0005, boot_entry_raw({0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x77, 0x00, 0x55, 0x00, 0x45, 0x00, 0x46, 0x00, 0x49, 0x00, 0x20, 0x00, 0x50,
	                                 0x00, 0x58, 0x00, 0x45, 0x00, 0x76, 0x00, 0x36, 0x00, 0x20, 0x00, 0x28, 0x00, 0x4d, 0x00, 0x41, 0x00, 0x43, 0x00, 0x3a, 0x00,
	                                 0x35, 0x00, 0x32, 0x00, 0x35, 0x00, 0x34, 0x00, 0x30, 0x00, 0x30, 0x00, 0x31, 0x00, 0x32, 0x00, 0x33, 0x00, 0x34, 0x00, 0x35,
	                                 0x00, 0x36, 0x00, 0x29, 0x00, 0x00, 0x00, 0x02, 0x01, 0x0c, 0x00, 0xd0, 0x41, 0x03, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01,
	                                 0x06, 0x00, 0x00, 0x03, 0x03, 0x0b, 0x25, 0x00, 0x52, 0x54, 0x00, 0x12, 0x34, 0x56, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	                                 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x03,
	                                 0x0d, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	                                 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	                                 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0x04, 0x00,
	                                 0x4e, 0xac, 0x08, 0x81, 0x11, 0x9f, 0x59, 0x4d, 0x85, 0x0e, 0xe2, 0x1a, 0x52, 0x2c, 0x59, 0xb2})},
	        // Boot0006-8be4df61-93ca-11d2-aa0d-00e098032b8c
	        {0x0006, boot_entry_raw({0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x5a, 0x00, 0x55, 0x00, 0x45, 0x00, 0x46, 0x00, 0x49, 0x00, 0x20, 0x00,
	                                 0x48, 0x00, 0x54, 0x00, 0x54, 0x00, 0x50, 0x00, 0x76, 0x00, 0x34, 0x00, 0x20, 0x00, 0x28, 0x00, 0x4d, 0x00, 0x41, 0x00,
	                                 0x43, 0x00, 0x3a, 0x00, 0x35, 0x00, 0x32, 0x00, 0x35, 0x00, 0x34, 0x00, 0x30, 0x00, 0x30, 0x00, 0x31, 0x00, 0x32, 0x00,
	                                 0x33, 0x00, 0x34, 0x00, 0x35, 0x00, 0x36, 0x00, 0x29, 0x00, 0x00, 0x00, 0x02, 0x01, 0x0c, 0x00, 0xd0, 0x41, 0x03, 0x0a,
	                                 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x06, 0x00, 0x00, 0x03, 0x03, 0x0b, 0x25, 0x00, 0x52, 0x54, 0x00, 0x12, 0x34, 0x56,
	                                 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	                                 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x03, 0x0c, 0x1b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	                                 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x18, 0x04, 0x00, 0x7f, 0xff,
	                                 0x04, 0x00, 0x4e, 0xac, 0x08, 0x81, 0x11, 0x9f, 0x59, 0x4d, 0x85, 0x0e, 0xe2, 0x1a, 0x52, 0x2c, 0x59, 0xb2})},
	        // Boot0007-8be4df61-93ca-11d2-aa0d-00e098032b8c
	        {0x0007,
	         boot_entry_raw({0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x7b, 0x00, 0x55, 0x00, 0x45, 0x00, 0x46, 0x00, 0x49, 0x00, 0x20, 0x00, 0x48, 0x00,
	                         0x54, 0x00, 0x54, 0x00, 0x50, 0x00, 0x76, 0x00, 0x36, 0x00, 0x20, 0x00, 0x28, 0x00, 0x4d, 0x00, 0x41, 0x00, 0x43, 0x00, 0x3a, 0x00,
	                         0x35, 0x00, 0x32, 0x00, 0x35, 0x00, 0x34, 0x00, 0x30, 0x00, 0x30, 0x00, 0x31, 0x00, 0x32, 0x00, 0x33, 0x00, 0x34, 0x00, 0x35, 0x00,
	                         0x36, 0x00, 0x29, 0x00, 0x00, 0x00, 0x02, 0x01, 0x0c, 0x00, 0xd0, 0x41, 0x03, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x06, 0x00,
	                         0x00, 0x03, 0x03, 0x0b, 0x25, 0x00, 0x52, 0x54, 0x00, 0x12, 0x34, 0x56, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	                         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x03, 0x0d, 0x3c, 0x00, 0x00,
	                         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	                         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00,
	                         0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x18, 0x04, 0x00, 0x7f, 0xff, 0x04, 0x00, 0x4e, 0xac, 0x08,
	                         0x81, 0x11, 0x9f, 0x59, 0x4d, 0x85, 0x0e, 0xe2, 0x1a, 0x52, 0x2c, 0x59, 0xb2})},
	        // Boot0008-8be4df61-93ca-11d2-aa0d-00e098032b8c
	        {0x0008, boot_entry_raw({0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x45, 0x00, 0x46, 0x00, 0x49, 0x00, 0x20, 0x00, 0x49,
	                                 0x00, 0x6e, 0x00, 0x74, 0x00, 0x65, 0x00, 0x72, 0x00, 0x6e, 0x00, 0x61, 0x00, 0x6c, 0x00, 0x20, 0x00, 0x53, 0x00,
	                                 0x68, 0x00, 0x65, 0x00, 0x6c, 0x00, 0x6c, 0x00, 0x00, 0x00, 0x04, 0x07, 0x14, 0x00, 0xc9, 0xbd, 0xb8, 0x7c, 0xeb,
	                                 0xf8, 0x34, 0x4f, 0xaa, 0xea, 0x3e, 0xe4, 0xaf, 0x65, 0x16, 0xa1, 0x04, 0x06, 0x14, 0x00, 0x83, 0xa5, 0x04, 0x7c,
	                                 0x3e, 0x9e, 0x1c, 0x4f, 0xad, 0x65, 0xe0, 0x52, 0x68, 0xd0, 0xb4, 0xd1, 0x7f, 0xff, 0x04, 0x00})},
	        // Boot0009-8be4df61-93ca-11d2-aa0d-00e098032b8c
	        {0x0009, boot_entry_raw({0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x62, 0x00, 0x64, 0x00, 0x65, 0x00, 0x62, 0x00, 0x69, 0x00, 0x61, 0x00, 0x6e,
	                                 0x00, 0x00, 0x00, 0x04, 0x01, 0x2a, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0,
	                                 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x95, 0x49, 0xd8, 0x6b, 0xa4, 0xb9, 0xa5, 0x47, 0xab, 0x45, 0x4d, 0xff, 0xfa, 0x6b, 0xe3,
	                                 0x5b, 0x02, 0x02, 0x04, 0x04, 0x34, 0x00, 0x5c, 0x00, 0x45, 0x00, 0x46, 0x00, 0x49, 0x00, 0x5c, 0x00, 0x64, 0x00, 0x65, 0x00,
	                                 0x62, 0x00, 0x69, 0x00, 0x61, 0x00, 0x6e, 0x00, 0x5c, 0x00, 0x73, 0x00, 0x68, 0x00, 0x69, 0x00, 0x6d, 0x00, 0x78, 0x00, 0x36,
	                                 0x00, 0x34, 0x00, 0x2e, 0x00, 0x65, 0x00, 0x66, 0x00, 0x69, 0x00, 0x00, 0x00, 0x7f, 0xff, 0x04, 0x00})},
	    },
	    {}  // no state
	};

	const klapki::state::state from_step_3 = [] {
		auto ret  = from_step_2;
		ret.order = klapki::state::boot_order_structured{.foreign = {0x0009, 0x0000, 0x0001, 0x0002, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007, 0x0008}};
		std::memcpy(ret.entries[0x0000].load_option_sha, "\x22\xa4\xf6\xee\x9a\xf6\xdb\xa0\x1d\x35\x28\xde\xb6\x4b\x74\xb5\x82\xfc\x18\x2b", sizeof(klapki::sha_t));
		std::memcpy(ret.entries[0x0001].load_option_sha, "\x0b\x5d\x2c\x98\xac\x5d\xe6\x14\x8a\x4a\x14\x90\xff\x9d\x5d\xf6\x90\x39\xf0\x4e", sizeof(klapki::sha_t));
		std::memcpy(ret.entries[0x0002].load_option_sha, "\x8f\x70\xc5\x6c\x75\xd0\x6c\x77\x71\x46\x85\x95\x73\xc2\xaa\x34\x37\x00\xc5\x52", sizeof(klapki::sha_t));
		std::memcpy(ret.entries[0x0003].load_option_sha, "\x20\xbd\x5f\x40\x22\x71\xd5\x7a\x88\xea\x31\x4f\xe3\x5c\x17\x05\x95\x6b\x1f\x74", sizeof(klapki::sha_t));
		std::memcpy(ret.entries[0x0004].load_option_sha, "\xf3\x4f\xff\xcd\x65\xe6\x1d\x17\xc7\x50\x97\x57\x1a\xfb\xcd\xcc\xd4\x3d\xd7\xf4", sizeof(klapki::sha_t));
		std::memcpy(ret.entries[0x0005].load_option_sha, "\xef\xfe\x79\x9f\x8a\x02\xbe\xea\x20\xa1\x53\x1d\xe0\x4e\xc5\xd9\x62\x6b\x5b\x51", sizeof(klapki::sha_t));
		std::memcpy(ret.entries[0x0006].load_option_sha, "\xb1\xe1\x37\xe8\x87\xd8\x06\xf1\x0c\xa7\xb6\xd7\xda\x83\x3c\xd9\x87\x18\x6e\x5e", sizeof(klapki::sha_t));
		std::memcpy(ret.entries[0x0007].load_option_sha, "\xec\x8c\x71\xc2\x11\x84\x02\x1f\xab\x57\xc3\x26\x87\x14\x5a\x17\x04\xcc\x02\x8c", sizeof(klapki::sha_t));
		std::memcpy(ret.entries[0x0008].load_option_sha, "\xdf\x5d\x66\x05\xcb\x8f\x43\x66\xd7\x45\xa8\x46\x4c\xfb\x26\xc1\xef\xdc\x30\x5c", sizeof(klapki::sha_t));
		std::memcpy(ret.entries[0x0009].load_option_sha, "\xba\xf8\x52\x2e\xf5\x77\x43\xec\x4e\xd5\xaa\x69\x21\xb3\xff\x83\x3d\x26\x3e\xc9", sizeof(klapki::sha_t));
		return ret;
	}();

	const klapki::context::context from_step_4{};

	const klapki::state::state from_step_5_state = [] {
		auto ret                   = from_step_3;
		ret.statecfg.boot_position = 4;
		return ret;
	}();
	const klapki::context::context from_step_5_context{.fresh_kernels = {{
	                                                       "version"sv,
	                                                       {"/"sv, "vmlinuz"sv},
	                                                       {{"/boot"sv, "initrd"sv}, {"/boot"sv, "initrd2"sv}, {"."sv, "initrd3"sv}},
	                                                   }}};

	const klapki::state::state from_step_6_state       = from_step_5_state;
	const klapki::context::context from_step_6_context = from_step_5_context;

	const klapki::state::state from_step_7_state = [] {
		auto ret = from_step_6_state;

		std::get<klapki::state::boot_order_structured>(ret.order).ours.emplace_back(0x000A);

		ret.entries[0x000A] = {.attributes = EFI_VARIABLE_NON_VOLATILE | EFI_VARIABLE_BOOTSERVICE_ACCESS | EFI_VARIABLE_RUNTIME_ACCESS};

		ret.statecfg.wanted_entries.emplace_back(klapki::state::stated_config_entry{
		    .bootnum_hint    = 0x000A,
		    .version         = "version",
		    .kernel_dirname  = "/",
		    .initrd_dirnames = {{"/boot", {}}, {std::nullopt, {}}, {"/tmp", {}}},
		});

		return ret;
	}();
	const klapki::context::context from_step_7_context = [] {
		auto ret                = from_step_6_context;
		ret.our_kernels[0x000A] = {"", "", {R"(\sr\version\)", "vmlinuz"}, {{{}, "initrd"}, {{}, "initrd2"}, {{}, "initrd3"}}};
		ret.fresh_kernels       = {};
		return ret;
	}();

	const klapki::context::context from_step_8 = [] {
		auto ret                            = from_step_7_context;
		ret.our_kernels[0x000A].description = "version"s;
		ret.our_kernels[0x000A].cmdline     = "root=PARTLABEL=klest"s;
		return ret;
	}();

	const klapki::state::state from_step_9_state = [] {
		auto ret = from_step_7_state;

		ret.order = klapki::state::boot_order_flat{
		    std::shared_ptr<std::uint16_t[]>{new std::uint16_t[]{0x0009, 0x0000, 0x0001, 0x0002, 0x000A, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007, 0x0008}}, 11};

		auto img = R"(\SR\VERSION\VMLINUZ)"sv;
		std::vector<std::uint8_t> devpath_file_node(efidp_make_file(nullptr, 0, const_cast<char *>(img.data())));
		assert(efidp_make_file(devpath_file_node.data(), devpath_file_node.size(), const_cast<char *>(img.data())) >= 0);

		efidp_data * devpath;
		assert(efidp_append_node(HD_Entire, reinterpret_cast<const efidp_data *>(devpath_file_node.data()), &devpath) >= 0);
		const auto devpath_len = efidp_size(devpath);

		auto cmdline_u8 = R"(initrd=\sr\version\initrd initrd=\sr\version\initrd2 initrd=\sr\version\initrd3 )"s += from_step_8.our_kernels.at(0x000A).cmdline;
		std::vector<std::uint16_t> cmdline;
		std::copy(std::begin(cmdline_u8), std::end(cmdline_u8), std::back_inserter(cmdline));

		ret.entries[0x000A].load_option_len =
		    efi_loadopt_create(nullptr, 0,                                                                                                    //
		                       ret.entries[0x000A].attributes,                                                                                //
		                       devpath, devpath_len,                                                                                          //
		                       reinterpret_cast<unsigned char *>(const_cast<char *>(from_step_8.our_kernels.at(0x000A).description.data())),  //
		                       reinterpret_cast<std::uint8_t *>(cmdline.data()), cmdline.size() * 2);

		ret.entries[0x000A].load_option = std::make_shared_for_overwrite<std::uint8_t[]>(ret.entries[0x000A].load_option_len);
		assert(efi_loadopt_create(ret.entries[0x000A].load_option.get(), ret.entries[0x000A].load_option_len,                                    //
		                          ret.entries[0x000A].attributes,                                                                                //
		                          devpath, devpath_len,                                                                                          //
		                          reinterpret_cast<unsigned char *>(const_cast<char *>(from_step_8.our_kernels.at(0x000A).description.data())),  //
		                          reinterpret_cast<std::uint8_t *>(cmdline.data()), cmdline.size() * 2) >= 0);

		std::memcpy(ret.entries[0x000A].load_option_sha, "\xfc\xfb\x67\xbc\x32\xc2\xbd\xce\x8e\xc5\x24\x3a\x70\xb7\x09\xe3\xf8\x9d\x2c\x60", sizeof(klapki::sha_t));
		std::memcpy(ret.statecfg.wanted_entries[0].load_option_sha, "\xfc\xfb\x67\xbc\x32\xc2\xbd\xce\x8e\xc5\x24\x3a\x70\xb7\x09\xe3\xf8\x9d\x2c\x60",
		            sizeof(klapki::sha_t));

		std::free(devpath);
		return ret;
	}();
	const klapki::context::context from_step_9_context = from_step_8;

	/// https://www.di-mgt.com.au/sha_testvectors.html
	static const std::string meg(1'000'000, 'a');
	std::vector<std::tuple<const char *, bool, std::string_view, std::string_view>> step_10_files{
	    {"/vmlinuz", false, "version"sv, "abc"sv},                                                           //
	    {"/boot/INITRD", false, "version"sv, "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"sv},  //
	    {"/boot/initrd2", false, "version"sv,
	     "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu"sv},  //
	    {"initrD3", false, "version"sv, meg}};
	const klapki::state::state from_step_10 = [] {
		auto ret = from_step_9_state;
		std::memcpy(ret.statecfg.wanted_entries[0].kernel_image_sha, "\xa9\x99\x3e\x36\x47\x06\x81\x6a\xba\x3e\x25\x71\x78\x50\xc2\x6c\x9c\xd0\xd8\x9d",
		            sizeof(klapki::sha_t));
		std::memcpy(ret.statecfg.wanted_entries[0].initrd_dirnames[0].second.data(),
		            "\x84\x98\x3e\x44\x1c\x3b\xd2\x6e\xba\xae\x4a\xa1\xf9\x51\x29\xe5\xe5\x46\x70\xf1", sizeof(klapki::sha_t));
		std::memcpy(ret.statecfg.wanted_entries[0].initrd_dirnames[1].second.data(),
		            "\xa4\x9b\x24\x46\xa0\x2c\x64\x5b\xf4\x19\xf9\x95\xb6\x70\x91\x25\x3a\x04\xa2\x59", sizeof(klapki::sha_t));
		std::memcpy(ret.statecfg.wanted_entries[0].initrd_dirnames[2].second.data(),
		            "\x34\xaa\x97\x3c\xd4\xc4\xda\xa4\xf6\x1e\xeb\x2b\xdb\xad\x27\x31\x65\x34\x01\x6f", sizeof(klapki::sha_t));
		return ret;
	}();
}
