// SPDX-License-Identifier: MIT


#include <cassert>
#include <doctest/doctest.h>
#include <optional>
#include <span>

#pragma GCC diagnostic ignored "-Wmissing-field-initializers"


namespace klapki::state {
	[[maybe_unused]]
	static bool operator==(const boot_entry & lhs, const boot_entry & rhs) {
		return std::basic_string_view<std::uint8_t>{lhs.load_option.get(), lhs.load_option_len} ==
		           std::basic_string_view<std::uint8_t>{rhs.load_option.get(), rhs.load_option_len} &&  //
		       !std::memcmp(lhs.load_option_sha, rhs.load_option_sha, sizeof(klapki::sha_t)) &&         //
		       lhs.attributes == rhs.attributes;
	}
}

namespace std {
	namespace {
		template <class E>
		bool operator==(const std::span<E> & lhs, const std::span<E> & rhs) {
			return lhs.size() == rhs.size() && std::equal(std::begin(lhs), std::end(lhs), std::begin(rhs));
		}
	}
}


namespace {
	struct err {
		std::optional<std::string> op;
		operator bool() { return !this->op; }
	};
}

namespace doctest {
	template <>
	struct StringMaker<err> {
		static String convert(const err & e) {
			using namespace std::literals;
			return e.op.value_or("{}"s).c_str();
		}
	};
}


namespace {
	efidp_data * HD_Entire = [] {
		efidp_data HD{.hd = {.header = {EFIDP_MEDIA_TYPE, EFIDP_MEDIA_HD, sizeof(efidp_hd)}}};
#ifdef INIT_HD
		INIT_HD
#else
		std::memcpy(HD.hd.signature, "faux klapki disk", sizeof(HD.hd.signature));
#endif
		const efidp_data Entire{.header = {EFIDP_END_TYPE, EFIDP_END_ENTIRE, sizeof(efidp_header)}};

		efidp_data *HD_ptr, *ret;
		assert(efidp_append_node(nullptr, &HD, &HD_ptr) == 0);
		assert(efidp_append_node(HD_ptr, &Entire, &ret) == 0);
		std::free(HD_ptr);
		return ret;
	}();
}


#define CONTEXT_EQ(yielded, from)                       \
	REQUIRE(yielded.our_kernels == from.our_kernels);     \
	REQUIRE(yielded.fresh_kernels == from.fresh_kernels); \
	REQUIRE(yielded.deleted_files == from.deleted_files);

#define STATE_EQ_STRUCTURED(yielded, from)                                                                                                              \
	REQUIRE(std::get<klapki::state::boot_order_structured>(yielded.order).ours == std::get<klapki::state::boot_order_structured>(from.order).ours);       \
	REQUIRE(std::get<klapki::state::boot_order_structured>(yielded.order).foreign == std::get<klapki::state::boot_order_structured>(from.order).foreign); \
	REQUIRE(yielded.entries == from.entries);                                                                                                             \
	REQUIRE(yielded.statecfg == from.statecfg);

#define FLAT_BOOT_ORDER_EQ(yielded, from)                                                             \
	const auto & yielded##_order = std::get<klapki::state::boot_order_flat>(yielded.order);             \
	const auto & from##_order    = std::get<klapki::state::boot_order_flat>(from.order);                \
	std::span<std::uint16_t> yielded##_order_s{yielded##_order.order.get(), yielded##_order.order_cnt}; \
	std::span<std::uint16_t> from##_order_s{from##_order.order.get(), from##_order.order_cnt};          \
	REQUIRE(yielded##_order_s.size() == from##_order_s.size());                                         \
	REQUIRE(std::equal(std::begin(yielded##_order_s), std::end(yielded##_order_s), std::begin(from##_order_s)))

#define STATE_EQ_FLAT(yielded, from)        \
	FLAT_BOOT_ORDER_EQ(yielded, from);        \
	REQUIRE(yielded.entries == from.entries); \
	REQUIRE(yielded.statecfg == from.statecfg);
