#!/bin/sh -fC
# SPDX-License-Identifier: 0BSD
# shellcheck disable=SC2059,SC2030,SC2031,SC2086,SC2188,SC2088

command -v gettext  > /dev/null || alias gettext='printf %s'
command -v ngettext > /dev/null || alias ngettext='printf %.0s%s'
niusage() {
	# The interval is between periodic events; the delay is between after the final occurrence of an event and it getting archived
	# next/last/when are the date equivalents – the date of the first recurrence, the date of the last occurrence, and the date of archival
	printf "$(TEXTDOMAIN=ratrun gettext 'non-interactive usage: %s [-Rv] [-r !|interval|next] [-m !|max|last] [-a !|delay|when] [-A archive] [%%]when event [reminders] [body…]\n                       %s [-Rv] [-r !|interval|next] [-m !|max|last] [-a !|delay|when] [-A archive] event [%%]when [reminders] [body…]\n')" "$0" "$0" >&2
	exit 1
}
usage() {
	printf "$(TEXTDOMAIN=ratrun gettext 'usage: %s [-Rv] [-r !|interval|next] [-m !|max|last] [-a !|delay|when] [-A archive] [[%%]when [event [reminders] [body…]]]\n       %s [-Rv] [-r !|interval|next] [-m !|max|last] [-a !|delay|when] [-A archive] [event [[%%]when [reminders] [body…]]]\n')" "$0" "$0" >&2
	exit 1
}

do_rerat=
rerat_verbose=
rerat_interval=
rerat_max=
rerat_archive=
rerat_archive_name=
while getopts 'Rvr:m:a:A:' arg; do
	case "$arg" in
		R)	do_rerat=1                               ;;
		v)	do_rerat=1; rerat_verbose=1              ;;
		r)	do_rerat=1; rerat_interval="$OPTARG"     ;;
		m)	do_rerat=1; rerat_max="$OPTARG"          ;;
		a)	do_rerat=1; rerat_archive="$OPTARG"      ;;
		A)	do_rerat=1; rerat_archive_name="$OPTARG" ;;
		*)	[ -t 0 ] && usage; niusage               ;;
	esac
done
shift $(( OPTIND - 1 ))

# shellcheck source=ratrun
. "${CONFDIR:-/etc/default}/ratrun"

[ -n "$do_rerat" ] && {
	[ -n "$rerat_interval" ]     && ! [ "$rerat_interval" = '!' ] && {
		ri="$(unsuff "$rerat_interval" '')"
		{ [ "${ri%[!0-9]*}" != "$ri" ] && ! date -d "$rerat_interval" +'' > /dev/null; } && { printf "$(TEXTDOMAIN=ratrun gettext '%s: -%c %s: must be !, a time interval, or a date\n')" "$0" 'r' "$rerat_interval" "$ri" >&2; exit 1; }
	}
	[ -n "$rerat_max" ]          && ! [ "$rerat_max" = '!' ] && { [ "${rerat_max%[!0-9]*}" != "$rerat_max" ] && ! date -d "$rerat_max" +'' > /dev/null; } &&
		{ printf "$(TEXTDOMAIN=ratrun gettext '%s: -m %s: must be !, an integer, or a date\n')" "$0" "$rerat_max" >&2; exit 1; }
	[ -n "$rerat_archive" ]      && ! [ "$rerat_archive" = '!' ] && {
		ra="$(unsuff "$rerat_archive" '')"
		{ [ "${ra%[!0-9]*}" != "$ra" ] && ! date -d "$rerat_archive" +'' > /dev/null; } && { printf "$(TEXTDOMAIN=ratrun gettext '%s: -%c %s: must be !, a time interval, or a date\n')" "$0" 'a' "$rerat_archive" "$ra" >&2; exit 1; }
	}
	[ -n "$rerat_archive_name" ] && [ "${rerat_archive_name%["$IFS"]*}" != "$rerat_archive_name" ] &&
		{ printf "$(TEXTDOMAIN=ratrun gettext '%s: -A %s: has blanks, unrepresentable (sorry)\n')" "$0" "$rerat_archive_name" >&2; exit 1; }
}

cd ~/.ratrun 2>&- || { mkdir -p ~/.ratrun && cd ~/.ratrun; } || exit
read -r TZ 2>/dev/null < .tz && export TZ


dfl=''

when="$dfl"; [ $# -ge 1 ] && { when="$1"; shift; }
evnt="$dfl"; [ $# -ge 1 ] && { evnt="$1"; shift; }
if   [ "$when" != "$dfl" ] && [ "$evnt" =  "$dfl" ] && ! date -d "${when#%}" +_ > /dev/null 2>&1; then
	printf "$(TEXTDOMAIN=ratrun gettext '%s: %s unacceptable as date, using as event name\n')" "$0" "$when" >&2
	tmp="$when"; when="$evnt"; evnt="$tmp"
elif [ "$when" != "$dfl" ] && [ "$evnt" != "$dfl" ] && ! date -d "${when#%}" +_ > /dev/null 2>&1 && date -d "${evnt#%}" +_ > /dev/null 2>&1; then
	printf "$(TEXTDOMAIN=ratrun gettext '%s: %s unacceptable as date, but %s is; exchanging\n')" "$0" "$when" "$evnt" >&2
	tmp="$when"; when="$evnt"; evnt="$tmp"
fi

dok=0
while :; do
	if [ "$when" = "$dfl" ]; then
		[ -t 0 ] || niusage
		TEXTDOMAIN=ratrun gettext 'Event date/time (start with % to canonicalise): ' >&2
		read -r when || exit
		# date(1) format here!
		date -d "${when#%}" +"${date_fmt:="$(TEXTDOMAIN=ratrun gettext 'Happens on %c')"}" || { when="$dfl"; continue; }
		dok=1
	else
		if ! date -d "${when#%}" +_ > /dev/null 2>&1; then
			printf "$(TEXTDOMAIN=ratrun gettext '%s: %s unacceptable as date\n')" "$0" "$when" >&2
			[ -t 0 ] && { when="$dfl"; continue; } || exit 1
		fi
	fi
	break
done
[ $dok -eq 1 ] || date -d "${when#%}" +"${date_fmt:="$(TEXTDOMAIN=ratrun gettext 'Happens on %c')"}"
whensecs="$(date -d "${when#%}" +'%s')"


read -r RERAT_DEFAULT 2>/dev/null < .rerat
parse_config $RERAT_DEFAULT
{ [ -n "$do_rerat" ] || [ -n "$all_events" ]; } && {
	[ -n "$rerat_interval" ]     && {
		rerat="${rerat_interval#!}"
		[ -n "$rerat" ] && {
			rerat="$(unsuff "$rerat" '')"
			[ "${rerat%[!0-9]*}" != "$rerat" ] && {
				rerat=$(( $(date -d "$rerat_interval" +'%s') - whensecs ))
				# argv[0]: -r {repeat date} (in human format): before {event date} (in human format)
				[ $rerat -lt 0 ] && { printf "$(TEXTDOMAIN=ratrun gettext '%s: -r %s (%s): before %s (%s)\n')" "$0" "$rerat_interval" "$(date -d "$rerat_interval" +'%c')" "${when#%}" "$(date -d "@$whensecs" +'%c')"; exit 1; }
				rerat_interval="$(resuff $rerat '')"
			}
		}
	}
	[ -n "$rerat_max"      ]     && {
		max="${rerat_max#!}"
		[ -n "$max" ] && {
			if [ "${max%[!0-9]*}" != "$max" ]; then
				# Happens if the final date was specified (-m), but time between events (-r) wasn't, so there's no way how to get to it
				[ -z "$rerat" ] && { printf "$(TEXTDOMAIN=ratrun gettext '%s: -m %s: no inter-event interval, can'\''t derive repeat count\n')" "$0" "$rerat_max"; exit 1; }
				max=$(( $(date -d "$max" +'%s') - whensecs ))
				[ $(( max % rerat )) -ne 0 ] && max=$(( max / rerat + 1 )) || max=$(( max / rerat ))
				# ((raw -m max date) - (raw when date)) / interval = repeats – (2022-03-04 - 2023-03-01) / 1d = -362, for example
				[ $max -le 0 ] && { printf "$(TEXTDOMAIN=ratrun gettext 'Max event schedules = ⌈(%s - %s) / %s⌉ = %d ≤ 0: nothing to do\n')" "$rerat_max" "${when#%}" "$rerat_interval" $max; exit; }
			else
				# As in "schedulings"; "to happen 0 times"
				[ "$max" -eq 0 ] && { TEXTDOMAIN=ratrun gettext 'Max event schedules = 0: nothing to do
'; exit; }
				max=$(( max - 1 ))
			fi
			rerat_max=$max
		}
	}

	last_date=$whensecs
	[ -n "$rerat" ] && {
		last_date=
		if [ -z "$max" ]; then
			# date(1) format here!
			date -d "@$(( whensecs + rerat ))" +"$(TEXTDOMAIN=ratrun gettext 'Will be re-scheduled ad infinitum; next time: %c')"
		elif [ "$max" -eq 0 ]; then
			# No rescheduling
			last_date=$whensecs
		elif [ "$max" -eq 1 ]; then
			last_date=$(( whensecs + rerat ))
			# date(1) format here!
			date -d "@$last_date" +"$(TEXTDOMAIN=ratrun gettext 'Will be re-scheduled once, for %c')"
		else
			last_date=$(( whensecs + rerat * max ))
			# n<=1 cases unreachable; times in %c format
			printf "$(TEXTDOMAIN=ratrun ngettext 'Will be re-scheduled %d time; next time: %s; last time: %s\n' 'Will be re-scheduled %d times; next time: %s; last time: %s\n' "$max")" "$max" "$(date -d "@$(( whensecs + $(unsuff "$rerat") ))" +'%c')" "$(date -d "@$last_date" +'%c')"
		fi
	}

	[ -n "$rerat_archive"  ]     && {  # this really is a repeat of the rerat_interval bit. no reference variables tho, ah well
		archive="${rerat_archive#!}"
		[ -n "$archive" ] && {
			archive="$(unsuff "$archive" '')"
			[ "${archive%[!0-9]*}" != "$archive" ] && {
				# rat: -a 2023-10-02: ...; error for when re-scheduling ad infinitum and -a is a date; "no date" as in "date doesn't exist"
				[ -z "$last_date" ] && { printf "$(TEXTDOMAIN=ratrun gettext '%s: -a %s: no final date, archive delay indeterminate\n')" "$0" "$rerat_archive"; exit 1; }
				archive=$(( $(date -d "$rerat_archive" +'%s') - last_date ))
				[ $archive -lt 0 ] && archive=0
				rerat_archive="$(resuff $archive '')"
			}
		}
	}
	[ -n "$rerat_archive_name" ] && { archive_name="$rerat_archive_name"; }

	[ -n "$archive" ] && {
		path="~/.ratrun/old/$archive_name"
		[ "${archive_name#/}" != "$archive_name" ] && path="$archive_name"
		[ -n "$last_date" ] &&
			# filename, date in %c format
			printf "$(TEXTDOMAIN=ratrun gettext 'Will be archived in %s on %s\n')" "$(date -d "@$last_date" +"$path")" "$(date -d "@$(( last_date + archive ))" +'%c')" ||
			# Written if re-scheduled ad infinitum, so will not be archived automatically
			printf "$(TEXTDOMAIN=ratrun gettext 'Would otherwise be archived in "%s"\n')" "$path"
	}

	[ -n "$rerat_max" ]          && [ -z "$rerat" ]   && printf "$(TEXTDOMAIN=ratrun gettext '%s: -m %s, but no interval configured: will not be re-scheduled in the current configuration\n')" "$0" "$rerat_max" >&2
	[ -n "$rerat_archive_name" ] && [ -z "$archive" ] && printf "$(TEXTDOMAIN=ratrun gettext '%s: -A %s, but no archive delay configured: will not be archived in the current configuration\n')" "$0" "$rerat_archive_name" >&2
}


while [ "$evnt" = "$dfl" ] || [ -z "$evnt" ]; do
	[ -t 0 ] || niusage
	TEXTDOMAIN=ratrun gettext 'Event: ' >&2
	read -r evnt || exit
done


cleanup() {
	printf "$(TEXTDOMAIN=ratrun gettext '%s: killed by signal, unscheduling %s\n')" "$0" "$nevnt" >&2
	rm -f -- "$nevnt"
	exit 127
}
trap cleanup HUP INT QUIT TERM
nevnt="$(
	trap 'rm -f -- "$nevnt"' HUP INT QUIT TERM
	[ "${evnt%/*}" != "$evnt" ] && evnt="$(printf '%s\n' "$evnt" | tr / _)"
	# empty file = no line = ignored
	nevnt="$evnt"                                              && > "$nevnt" && printf '%s\n' "$nevnt" && exit
	nevnt="$evnt-${when#%}"                                    && > "$nevnt" && printf '%s\n' "$nevnt" && exit
	nevnt="$evnt-$(date -d "@$whensecs" +'%Y-%m-%dT%H:%M%:z')" && > "$nevnt" && printf '%s\n' "$nevnt" && exit  # -Iminutes
)" 2>/dev/null
[ -z "$nevnt" ] && {
	# User-entered date/time here; likely to be a time
	printf "$(TEXTDOMAIN=ratrun gettext "%s: too many identical events, couldn't allocate file for \"%s\" at %s\n")" "$0" "$evnt" "${when#%}" >&2
	exit 1
}
# "renamed to remove collisions"
[ "$nevnt" != "$evnt" ] && printf "$(TEXTDOMAIN=ratrun gettext '%s: "%s" decollided to "%s"\n')" "$0" "$evnt" "$nevnt" >&2
evnt="$nevnt"

exec >> "$evnt"
flock 1

if [ "${when#%}" != "$when" ]; then
	date -d "@$whensecs" +'%Y-%m-%dT%H:%M%:z'  # -Iminutes
else
	printf '%s\n' "$when"
fi


rmnd="$dfl"; [ $# -ge 1 ] && { [ "${1#[0-9]}" != "$1" ] && { rmnd="$1"; shift; } || rmnd=; }

demind() {
	ret=0
	for r; do
		secs="$(unsuff "$r" '')"
		if [ "${secs%[!0-9]*}" != "$secs" ]; then
			onreflow
			secs="$(date -d "$r" +'%s')" || { onerr; continue; }
			secs=$(( whensecs - secs ))
			# user input, then unsuffed reminder ("20h"/"118m")
			[ "$secs" -ge 0 ] || { printf "$(TEXTDOMAIN=ratrun gettext '%s: %s is %safter the event!\n')" "$0" "$r" "$(resuff "${secs#-}")" >&2; onerr; continue; }
		fi
		resuff "$secs"
	done
	exit $ret
}

read -r RATRUN_REMINDERS 2>/dev/null < .reminders

# In context of reminders; used as the default value below instead of a list
[ -z "$RATRUN_REMINDERS" ] && RATRUN_REMINDERS="$(TEXTDOMAIN=ratrun gettext 'none!')"


reencoded=
[ "$rmnd" = "$dfl" ] && {
	if [ -e '.nevermind' ]; then
		rmnd=
	elif [ -t 0 ]; then
		while [ "$rmnd" = "$dfl" ]; do
			printf "${remp:="$(TEXTDOMAIN=ratrun gettext 'Reminders (clock times OK; empty = default (%s)): ')"}" "$RATRUN_REMINDERS" >&2
			read -r rmnd || {
				exec <&-  # Might want to turn this into < /dev/null if we ever prompt for anything more than a body
				rmnd=
				# User ^Ded in the reminders prompt, this is to inform them that an empty mail body will be used for the event notifications; "Using empty body"/"No body"
				printf "$(TEXTDOMAIN=ratrun gettext '\nBody empty.\n')" >&2
				break
			}

			rmnd="$(onreflow() { ret=66; }; onerr() { exit 1; }; demind $rmnd)"
			case $? in
				0)	reencoded=y ;;
				# In context of the reminders, e.g. if you said 18:45 this may say 15m
				66)	reencoded=y; printf "$(TEXTDOMAIN=ratrun gettext 'Reencoded as %s\n')" "$rmnd" >&2 ;;
				*)	rmnd="$dfl" ;;
			esac
			rmnd="${rmnd% }"
		done
	else
		rmnd=
	fi
}

[ -n "$rmnd" ] && [ -z "$reencoded" ] && {
	nr="$(onreflow() { :; }; onerr() { printf '%s ' "$r"; ret=1; }; demind $rmnd)" || {
    # In context of the reminders, e.g. if you said 18: (which is invalid); actual times listed by date(1)
		printf "$(TEXTDOMAIN=ratrun gettext '%s: using the above verbatim\n')" "$0" >&2
	}
	nr="${nr% }"
	# In context of the reminders, e.g. if you said 18:45
	[ "$nr" != "$rmnd" ] && printf "$(TEXTDOMAIN=ratrun gettext '%s: reminders normalised to %s\n')" "$0" "$nr" >&2
	rmnd="$nr"
}

printf '%s\n' "$rmnd"


if [ $# -ge 1 ]; then
	printf '%s\n' "$*"
elif [ -t 0 ]; then
	printf "$(TEXTDOMAIN=ratrun gettext 'Enter body; %s to finish\n')" "$(stty -a | sed -n 's/;/\n/g;/^[[:space:]]*eof[[:space:]]*=[[:space:]]*/ {s///;P;q};D')" >&2
	cat
else
	: No body
fi


[ -n "$do_rerat" ] && {
	printf '\nrerat:'
	[ -n "$rerat_verbose" ]      && printf ' %s' "$rerat_verbose"
	[ -n "$rerat_interval" ]     && printf ' %s' "$rerat_interval"
	[ -n "$rerat_max" ]          && printf ' max=%s' "$rerat_max"
	[ -n "$rerat_archive" ]      && printf ' archive=%s' "$rerat_archive"
	[ -n "$rerat_archive_name" ] && printf ' archive-name=%s' "$rerat_archive_name"
	printf '\n'
}
