#!/bin/sh -f
# SPDX-License-Identifier: 0BSD
# shellcheck disable=SC2086,SC2046,SC2059,SC2019,SC2016,SC2188

# This whole prologue, past -a processing (sans CONFDIR fallbacks, -d .ratrun/old, and mail subject), is shared with rerat
command -v gettext > /dev/null || alias gettext='printf %s'
usage() {
	printf "$(TEXTDOMAIN=ratrun gettext 'usage: %s [-n[n]] [-d date]\n       %s -a\n')" "$0" "$0" >&2
	exit 1
}
all=
noop=
now=now
while getopts 'and:' arg; do
	case "$arg" in
		a)	all=a                      ;;
		n)	noop=$(( ${noop:-0} + 1 )) ;;
		d)	now="$OPTARG"              ;;
		*)	usage                      ;;
	esac
done
shift $(( OPTIND - 1 ))
{ [ $# -gt 0 ] || { [ -n "$all" ] && { [ -n "$noop" ] || ! [ "$now" = 'now' ]; }; }; } && usage

# shellcheck source=ratrun
. "${CONFDIR:-/etc/default}/ratrun"

[ -n "$all" ] && {
	getent passwd $({
		printf '%s\n' $RATRUN_USERS
		[ -n "$RATRUN_GROUPS" ] && getent group $RATRUN_GROUPS | while IFS=: read -r _ _ _ users _; do IFS=,; printf '%s\n' $users; done
	} | LC_ALL=C sort -u) | {
		while IFS=: read -r username _ uid gid _ homedir _; do
			[ -d "$homedir/.ratrun" ] || continue
			HOME="$homedir" setpriv --reuid="$uid" --regid="$gid" --init-groups --inh-caps=-all "$0" 2>&1 |
				mail_noempty -s "$(printf "$(TEXTDOMAIN=ratrun gettext 'Errors for your ratrun at %s')" "$(date)")" "$username" &
		done
		wait
	}
	exit
}


[ -n "$noop" ] && {
	noop_mail() { printf '%s\n' "mail $*"; [ $noop -ge 2 ] && cat && echo '-- >8 --'; false; }
	alias mail=noop_mail
	alias mkdir=false
}

cd ~/.ratrun 2>&- || exit 0
read -r RATRUN_REMINDERS 2>/dev/null < .reminders
read -r TZ               2>/dev/null < .tz        && export TZ
read -r ratpref          2>/dev/null < .prefix    || ratpref='ratrun:'
now="$(date -d "$now" +'%s')" || exit
username="$(id -un)"
cr="$(printf '\r')"  # this wants to be literal in the only place that uses it, but https://github.com/sublimehq/sublime_text/issues/5646

# The header parsing is mirrored in rerat do_file() very closely
set +f
for f in *; do
	set -f

	{ ! [ -e "$f" ] || [ -d "$f" ]; } && continue
	{ < "$f"; } 2>&- || continue
	exec < "$f"
	flock -n 0 || continue

	alias cleanup > /dev/null 2>&1 && eval cleanup && trap - EXIT && unalias cleanup

	evname="$f"
	read -r rawtime || continue  # empty/non-text file
	rawtime="${rawtime%"$cr"}"
	if [ "$rawtime" = 'BEGIN:VCALENDAR' ]; then
		alias cleanup='rm -f -- "$tmpf"'
		trap cleanup EXIT
		# An error is output by mktemp(1), this is an annotation
		tmpf="$(mktemp -t ratrun.XXXXXXXXXX)" || { printf "$(TEXTDOMAIN=ratrun gettext '%s: for iCalendar in %s\n')" "$0" "$f" >&2; continue; }


		# See also rerat.sh for a cut-down version
		# Summary (against baseline of 13.2±1.5ms):
		#   'cat * * * * * | sed -ne '1{h;n}' -e H -e '${x;s/\n[  ]//g;p}'' ran
		#     1.57 ± 0.21 times faster than 'cat * * * * * | awk '/^[ \t]/ { ps = ps substr($0, 2); next }  {if(ps) print ps; ps = $0}  END {if(ps) print ps}''
		#     4.99 ± 0.60 times faster than 'cat * * * * * | mawk '/^[ \t]/ { ps = ps substr($0, 2); next }  {if(ps) print ps; ps = $0}  END {if(ps) print ps}''
		#   194.15 ± 21.04 times faster than 'cat * * * * * | busybox sed -ne '1{h;n}' -e H -e '${x;s/\n[  ]//g;p}''
		# This is additionally pared down to not be equivalent (blank line inserted at the top and not removed from the body)
		sed -n 'H;${x;s/\n[ 	]//g;p}' |
			# argv[0]: filename: TRIGGER:-PT30M,-PT30D,-P7W,-P15DT5H0M20S: "-PT30D" invalid, "30D" left
			awk -v alarmerr="${alarmerr:="$(TEXTDOMAIN=ratrun gettext '%s: %s: %s: "%s" invalid, "%s" left\n')"}" -v argv0="$0" -v filename="$f" '
				BEGIN                             { inevent=0 }
				/^(X-WR-TIMEZONE|TZID)(;[^:]+)?:/ { tz=$0; sub(/^[^:;]+(;[^:]+)?:/, "", tz) }
				/^BEGIN:VEVENT/                   { inevent=1; evtz=summary=start=reminders=description="" }
				inevent && /^SUMMARY(;[^:]+)?:/   { summary=$0; sub(/^SUMMARY(;[^:]+)?:/, "", summary) }
				inevent && /^DTSTART(;[^:]+)?:/   {
					start=$0
					sub(/^DTSTART(;[^:]+)?:/, "", start)
					sub(/^[0-9][0-9][0-9][0-9]/, "&-", start)
					sub(/^[0-9][0-9][0-9][0-9]-[0-9][0-9]/, "&-", start)
					sub(/^[0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]/, "&:", start)
					sub(/^[0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]:[0-9][0-9]/, "&:", start)

					maybetz=$0
					if(index(";", maybetz)) {
						sub(/:[^:]+$/, "", maybetz)
						sub(/^[^;]+;/, "", maybetz)
						split(maybetz, maybetz_arr, ";")
						for(maybetz in maybetz_arr)
							if(sub(/^TZID=/, "",  maybetz_arr[maybetz])) {
								evtz=maybetz_arr[maybetz]
								sub(/^"/, "", evtz)
								sub(/"$/, "", evtz)
								next
							}
					}
				}
				/^X-RATSTART:/                    {
					start=$0
					sub(/^X-RATSTART:/, "", start)
				}
				# mandatory part of BEGIN:VALARM, all other fields are fully bunk in practice
				# Assumes all are relative to the start, and ignores the absolute ones
				inevent && /^TRIGGER(;[^:]+)?:/ {
					r=$0
					sub(/^TRIGGER(;[^:]+)?:/, "", r)

					# https://www.rfc-editor.org/rfc/rfc5545#section-3.3.6
					# only accept negative (to-the-past-looking)
					# > If the property permits, multiple "duration" values are specified by a COMMA-separated list of values.
					# I havent seen this, but its supported, if it looks like I think it does!
					split(r, r_arr, ",")
					for(r in r_arr) {
						rf=r=r_arr[r]

						if(!sub(/^-P/, "", r))
							continue

						# Actually a superset of the specified (dur-week is allowed to be followed by dur-date)
						racc=0
						if(match(r, /^[0-9]+W/)) {
							racc += substr(r, 1, RLENGTH) * 7 * 24 * 60 * 60
							r=substr(r, RLENGTH + 1)
						}
						if(match(r, /^[0-9]+D/)) {
							racc += substr(r, 1, RLENGTH) * 24 * 60 * 60
							r=substr(r, RLENGTH + 1)
						}
						if(sub(/^T/, "", r)) {
							if(match(r, /^[0-9]+H/)) {
								racc += substr(r, 1, RLENGTH) * 60 * 60
								r=substr(r, RLENGTH + 1)
							}
							if(match(r, /^[0-9]+M/)) {
								racc += substr(r, 1, RLENGTH) * 60
								r=substr(r, RLENGTH + 1)
							}
							if(match(r, /^[0-9]+S/)) {
								racc += substr(r, 1, RLENGTH)
								r=substr(r, RLENGTH + 1)
							}
						}

						if(r) {
							printf alarmerr, argv0, filename, $0, rf, r > "/dev/stderr"
							continue
						}

						reminders=reminders " " racc
					}
				}
				inevent && /^DESCRIPTION(;[^:]+)?:/ { description=$0; sub(/^DESCRIPTION(;[^:]+)?:/, "", description) }
				# TODO? handle multiple events gracefully, maybe;
				# out of the entire corpus not one interactive thing allowed to export/send more than one at a time,
				# the standard is explicitly sceptical of this, and the only thing that even has it is an automated ticket system (1 event per ticket)
				/^END:VEVENT/ && summary && start {
					if(!evtz)
						evtz=tz

					# Format enforcement
					gsub(/\\n/, " ", evtz); gsub(/\\n/, " ", summary); gsub(/\\n/, " ", start)
					# https://www.rfc-editor.org/rfc/rfc5545#section-3.3.11
					gsub(/\\;/, ";", evtz); gsub(/\\;/, ";", summary); gsub(/\\;/, ";", start); gsub(/\\;/, ";", description)
					gsub(/\\,/, ",", evtz); gsub(/\\,/, ",", summary); gsub(/\\,/, ",", start); gsub(/\\,/, ",", description)

					print evtz
					print summary
					print start
					print reminders
					print description

					inevent=0
				}
			' | {
				while read -r l; do printf '%b\n' "$l"; done
				printf '\n-- \n~/.ratrun/%s\n' "$f"
			} > "$tmpf"

		exec 4<&0 < "$tmpf"
		eval cleanup
		unzone_e=
		alias cleanup='exec 4<&-; unset TZ; eval "$unzone_e"'

		guess_tz
		read -r evname
		read -r rawtime
	fi

	# An error is output by date(1), this is an annotation
	filetime=$(date -d "$rawtime" +'%s') || { printf "$(TEXTDOMAIN=ratrun gettext '%s: in file %s\n')" "$0" "$f" >&2; continue; }

	ungetl=
	read -r reminders && [ "${reminders#[0-9]}" = "$reminders" ] && {
		ungetl="$reminders"
		reminders=
	}
	[ -z "$reminders" ] && reminders="$RATRUN_REMINDERS"
	[ -z "$reminders" ] && continue
	reminders="$(for r in $reminders; do unsuff "$r"; done)"
	reminders_cnt=0; for r in $reminders; do reminders_cnt=$(( reminders_cnt + 1 )); done
	[ $reminders_cnt -eq 0 ] && continue

	timeleft=$(( filetime - now ))
	expired_least=
	expired_cnt=0
	unexpired=
	for r in $reminders; do
		if [ $timeleft -le $r ]; then
			{ [ -z "$expired_least" ] || [ $(( timeleft - r )) -gt $expired_least ]; } && expired_least=$r
			expired_cnt=$(( expired_cnt + 1 ))
		else
			unexpired=1
		fi
	done

	read -r prev_expired_cnt 2>/dev/null < ".expcnt/$f" || prev_expired_cnt=0

	do_expcnt() { mkdir -p '.expcnt' && echo $expired_cnt > ".expcnt/$f"; }
	[ $expired_cnt -gt "$prev_expired_cnt" ] && {
		send() {
			fulldt="$(date -d"@$filetime" +'%R')"
			if [ "$fulldt" = "$rawtime" ] || [ "${fulldt#0}" = "$rawtime" ]; then
				subj_fmt="${subj_fmt_s:="$(TEXTDOMAIN=ratrun gettext '%s in %s / on %s%.0s: %s')"}"
			else
				subj_fmt="${subj_fmt_l:="$(TEXTDOMAIN=ratrun gettext '%s in %s / on %s (%s): %s')"}"
			fi
			mail -s "$(printf "$subj_fmt" "$ratpref" "$(resuff "$expired_least" '')" "$fulldt" "$rawtime" "$evname")" "$username"
		}
		if [ -n "$ungetl" ]; then
			{ printf '%s\n' "$ungetl"; cat; } | send  # Piping to subshell means we won't propagate caches of subj_fmt_[sl] upward; ah well
		else
			send
		fi && [ $expired_cnt -lt $reminders_cnt ] && do_expcnt
	}

	[ -z "$unexpired" ] && {
		mkdir -p 'old' || {
			do_expcnt
			continue
		}
		# yes, this nominally races between the datemove and the rm: there's no way in hell other processing could inject, process, and send a mail in that time tho
		# yes, this overrides old/$f-$(date -Iminutes); realistically this is not an issue,
		#                                               even if you schedule dupa for 12:00 on multiple days they get moved to different files.
		datemove "$f" "old/" "$rawtime" "$filetime" && rm -f ".expcnt/$f"
	}
done

cd '.expcnt' 2>/dev/null && {
	set +f
	for f in *; do
		set -f

		[ -e "../$f" ] || rm -f -- "$f" 2>/dev/null
	done

	rm -fd "$PWD" 2>/dev/null
}
:
