#!/bin/sh -f
# SPDX-License-Identifier: 0BSD
# shellcheck disable=SC2086,SC2088,SC2046,SC2059,SC2016,SC2188

# This whole prologue, past -a processing (sans CONFDIR fallbacks, -d .ratrun/old, and mail subject), is shared with ratrun
command -v gettext > /dev/null || alias gettext='printf %s'
usage() {
	printf "$(TEXTDOMAIN=ratrun gettext 'usage: %s [-n[n]] [-d date]\n       %s -a\n')" "$0" "$0" >&2
	exit 1
}
all=
noop=
now=now
while getopts 'and:' arg; do
	case "$arg" in
		a)	all=a                      ;;
		n)	noop=$(( ${noop:-0} + 1 )) ;;
		d)	now="$OPTARG"              ;;
		*)	usage                      ;;
	esac
done
shift $(( OPTIND - 1 ))
{ [ $# -gt 0 ] || { [ -n "$all" ] && { [ -n "$noop" ] || ! [ "$now" = 'now' ]; }; }; } && usage

# We can't reject take-off in a datemove() polyfill, since ed has already edded
# shellcheck source=ratrun
. "${CONFDIR:-/etc/default}/ratrun"

[ -n "$all" ] && {
	getent passwd $({
		printf '%s\n' $RATRUN_USERS
		[ -n "$RATRUN_GROUPS" ] && getent group $RATRUN_GROUPS | while IFS=: read -r _ _ _ users _; do IFS=,; printf '%s\n' $users; done
	} | LC_ALL=C sort -u) | {
		while IFS=: read -r username _ uid gid _ homedir _; do
			[ -d "$homedir/.ratrun/old" ] || continue
			HOME="$homedir" setpriv --reuid="$uid" --regid="$gid" --init-groups --inh-caps=-all "$0" 2>&1 |
				mail_noempty -s "$(printf "$(TEXTDOMAIN=ratrun gettext 'Notes from your rerat run at %s')" "$(date)")" "$username" &
		done
		wait
	}
	exit
}


[ -n "$noop" ] && {
	noop_archive() { printf 'tar -rf %s %s\n' "$1" "$evname"; false; }
	alias archive=noop_archive

	noop_ed() { printf '%s: %s + %s -> %s\n' "$evname" "$rawtime" "$rerat" "$new_rawtime"; [ $noop -ge 2 ] && printf 'ed %s\n' "$3" && cat && echo '-- >8 --'; false; }
	alias ed=noop_ed
}

cd ~/.ratrun/old 2>&- || exit 0
read -r RERAT_DEFAULT 2>/dev/null < ../.rerat
read -r TZ            2>/dev/null < ../.tz    && export TZ
now="$(date -d "$now" +'%s')" || exit
cr="$(printf '\r')"  # this wants to be literal in the only place that uses it, but https://github.com/sublimehq/sublime_text/issues/5646


# The header parsing mirrors the main loop in ratrun very closely
do_file() {
	{ ! [ -e "$f" ] || [ -d "$f" ]; } && return
	{ < "$f"; } 2>&- || return
	exec < "$f"
	flock -n 0 || return

	alias cleanup > /dev/null 2>&1 && eval cleanup && trap - EXIT && unalias cleanup
	unset ical

	evname="$f"
	read -r rawtime || return  # empty/non-text file
	rawtime="${rawtime%"$cr"}"
	if [ "$rawtime" = 'BEGIN:VCALENDAR' ]; then
		alias cleanup='rm -f -- "$tmpf"'
		trap cleanup EXIT
		# An error is output by mktemp(1), this is an annotation
		tmpf="$(mktemp -t rerat.XXXXXXXXXX)" || { printf "$(TEXTDOMAIN=ratrun gettext '%s: for iCalendar in %s\n')" "$0" "$f" >&2; return; }


		# Agony. TODO: the big blocker here is that BYSETPOS is unimplementable (too me). or at least I can't come up with a reasonable approach for it
		# RRULE:FREQ=DAILY;UNTIL=20230522T220000Z;INTERVAL=2                       ICSy-od-starego+.d/__calendar-chrustening.ics
		# RRULE:FREQ=MONTHLY;BYDAY=WE;BYSETPOS=-1                                  ICSy-od-starego+.d/__mies3.ics
		# RRULE:FREQ=MONTHLY;BYMONTHDAY=-1                                         ICSy-od-starego+.d/__eventodliczyć.ics
		# RRULE:FREQ=MONTHLY;COUNT=11;INTERVAL=2;BYMONTHDAY=9                      ICSy-od-starego+.d/__mies1.ics
		# RRULE:FREQ=MONTHLY;INTERVAL=2;BYDAY=TH;BYSETPOS=3                        ICSy-od-starego+.d/__mies2.ics
		# RRULE:FREQ=MONTHLY;UNTIL=20230524T070000Z;BYDAY=TH;BYSETPOS=-1           ICSy-od-starego+.d/__calendar-lastofmonth.ics
		# RRULE:FREQ=WEEKLY;BYDAY=MO,TU,WE,TH,FR                                   ICSy-od-starego+.d/__dzienpowszoniec-cancelled.ics
		# RRULE:FREQ=WEEKLY;BYDAY=SU,MO,WE,TH                                      ICSy-od-starego+.d/__tygod.ics
		# RRULE:FREQ=WEEKLY;UNTIL=20230105T090000Z;INTERVAL=1;BYDAY=TH;WKST=SU     ICSy-od-starego+.d/__PSBTranslations-progressreview&planning2p
		# RRULE:FREQ=YEARLY;BYDAY=TU;BYMONTH=4;BYSETPOS=3                          ICSy-od-starego+.d/__rok2.ics
		# RRULE:FREQ=YEARLY;INTERVAL=2;BYMONTHDAY=4;BYMONTH=1                      ICSy-od-starego+.d/__rok1.ics
		# RRULE:FREQ=WEEKLY;COUNT=3;BYDAY=WE                                       ICSy-od-starego+.d/__calendar-wednesday.ics
		# RRULE;X-EVOLUTION-ENDDATE=20230524T000000Z:FREQ=WEEKLY;COUNT=3;BYDAY=WE  ICSy-od-starego+.d/__calendar-wednesday-cancelled.ics
		# (there's also RDATE, which no-one uses; as well as EXDATE, which..? why would you)
		# (multiple RRULEs can be specced and the total events are the sum. you literally can't make that be the case interactively)

		# We're making the grave assumption that DTSTART and RRULE are always one-line; this holds for my test dataset :)
		# This mirrors the approach in ratrun loosely (marked # ratrun); we don't care about as many things and we process this differently, however
		sed -n "s/$cr//g;"'H;${x;s/\n[ 	]//g;p}' |
			# TRIGGER, DESCRIPTION removed; otherwise same as ratrun
			awk '
				BEGIN                             { inevent=ratstarts=0; additional="" }
				/^(X-WR-TIMEZONE|TZID)(;[^:]+)?:/ { tz=$0; sub(/^[^:;]+(;[^:]+)?:/, "", tz) }
				/^BEGIN:VEVENT/                   { inevent=1; evtz=summary=start="" }
				inevent && /^SUMMARY(;[^:]+)?:/   { summary=$0; sub(/^SUMMARY(;[^:]+)?:/, "", summary) }
				inevent && /^DTSTART(;[^:]+)?:/   {
					start=$0
					sub(/^DTSTART(;[^:]+)?:/, "", start)
					sub(/^[0-9][0-9][0-9][0-9]/, "&-", start)
					sub(/^[0-9][0-9][0-9][0-9]-[0-9][0-9]/, "&-", start)
					sub(/^[0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]/, "&:", start)
					sub(/^[0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]:[0-9][0-9]/, "&:", start)

					maybetz=$0
					if(index(";", maybetz)) {
						sub(/:[^:]+$/, "", maybetz)
						sub(/^[^;]+;/, "", maybetz)
						split(maybetz, maybetz_arr, ";")
						for(maybetz in maybetz_arr)
							if(sub(/^TZID=/, "",  maybetz_arr[maybetz])) {
								evtz=maybetz_arr[maybetz]
								sub(/^"/, "", evtz)
								sub(/"$/, "", evtz)
								next
							}
					}
				}
				/^X-RATSTART:/                    {
					start=$0
					sub(/^X-RATSTART:/, "", start)
					++ratstarts
				}
				# see note in ratrun for multi-event
				/^END:VEVENT/ && summary && start {
					if(!evtz)
						evtz=tz

					# Format enforcement
					gsub(/\\n/, " ", evtz); gsub(/\\n/, " ", summary); gsub(/\\n/, " ", start)
					# https://www.rfc-editor.org/rfc/rfc5545#section-3.3.11
					gsub(/\\;/, ";", evtz); gsub(/\\;/, ";", summary); gsub(/\\;/, ";", start)
					gsub(/\\,/, ",", evtz); gsub(/\\,/, ",", summary); gsub(/\\,/, ",", start)

					print evtz
					print summary
					print start

					inevent=0
				}
				/^rerat/                          { additional=additional "\n" $0 }
				END                               {
					gsub("\\\\", "\\\\", additional)
					print additional
					print "rerat: reratted=" ratstarts
				}
			' | while read -r l; do printf '%b\n' "$l"; done > "$tmpf"

		exec 4<&0 < "$tmpf"
		eval cleanup
		unzone_e=
		alias cleanup='exec 4<&-; unset TZ; eval "$unzone_e"'

		ical=1
		guess_tz
		read -r evname
		read -r rawtime
	fi


	unset rerat max reratted archive archive_name verbose rerat_lineno reratted_lineno last_lineno anysep
	original_time=" original-time=$rawtime"
	lineno=1
	while read -r line; do
		lineno=$(( lineno + 1 ))
		{ [ "${line#rerat["$IFS"]:}" != "$line" ] || [ "${line#rerat:}" != "$line" ]; } && { parse_config ${line#rerat*:}; last_lineno=$lineno; }
	done

	[ -z "${rerat-"$default_rerat"}${archive-"$default_archive"}" ] && return
	filetime=$(date -d "$rawtime" +'%s') || { printf "$(TEXTDOMAIN=ratrun gettext '%s: in file %s\n')" "$0" "$f" >&2; return; }


	[ -n "${rerat-"$default_rerat"}" ] && {
		rerat_secs="$(unsuff "${rerat-"$default_rerat"}" '')"
		[ "${rerat_secs%[!0-9]*}" != "$rerat_secs" ] &&
			{ printf "$(TEXTDOMAIN=ratrun gettext '%s: %s: rerat interval %s invalid\n')" "$0" "$f" "${rerat-"$default_rerat"}" >&2; return; }

		new_filetime=$(( filetime + rerat_secs ))
		new_rawtime="$(date -d"@$new_filetime" +'%Y-%m-%dT%H:%M%:z')"  # -Iminutes
		[ -n "$reratted_lineno" ] && { [ "${reratted%[!0-9]*}" != "$reratted" ] || [ -z "$reratted" ]; } &&
			# An error is output by the shell, this is an annotation; "assuming/[treating as-if] post has been reratted only once previously"
			{ (: $(( "$reratted" + 1 ))); printf "$(TEXTDOMAIN=ratrun gettext '%s: in file %s; assuming 1\n')" "$0" "$f" >&2; reratted=1; }

		[ -n "${max-"${default_max}"}" ] && [ "${reratted:-0}" -ge "${max-"${default_max}"}" ] && rerat=
	}
	[ -n "${rerat-"$default_rerat"}" ] && {
		if [ -z "$ical" ]; then
			for sep in '/' ';' '|' '@' ',' ':'; do
				[ "${original_time%"$sep"*}" = "$original_time" ] && anysep=1 && break
			done
			# argv[0]: filename: raw date
			[ -n "$anysep" ] || { printf "$(TEXTDOMAIN=ratrun gettext '%s: %s:%s: pathological (contains /;|@,:); not saving\n')" "$0" "$f" "$original_time" >&2; original_time=; }

			printf '%s\n' '1d' i "$new_rawtime" .
			[ -n "$reratted_lineno" ] && printf '%s\n' "${reratted_lineno}s${sep}reratted=[^[:space:]]*${sep}reratted=$(( reratted + 1 ))$original_time${sep}" ||
				{ [ -n "${rerat_lineno-$last_lineno}" ] && printf '%s\n' "${reratted_lineno-$last_lineno}s${sep}[[:space:]]*\$${sep} reratted=1$original_time${sep}"; } ||
				printf '%s\n' '$a' "rerat: reratted=1$original_time" .
			printf '%s\n' w
		else
			printf '%s\n' '/END:VEVENT/i' "X-RATSTART:$new_rawtime" . w
		fi | ed -s -- "$f" && datemove "$f" '../' "$new_rawtime" "$new_filetime" &&  # TODO? these are the same
			[ -n "${verbose-"${default_verbose}"}" ] &&
			# the time here is in "natural" (date %c) format
			printf "${verb_resc_fmt:="$(TEXTDOMAIN=ratrun gettext 'Event "%s" re-scheduled for %s\n')"}" "$evname" "$(date -d"@$new_filetime" +'%c')"
		return
	}


	timepast=$(( now - filetime ))
	[ -n "${archive-"$default_archive"}" ] && {
		archive="$(unsuff "${archive-"$default_archive"}" '')"
		[ $timepast -ge "$archive" ] && {
			arch_f="$(date -d"@$filetime" +"${archive_name-"${default_archive_name}"}")"
			archive "$arch_f" "$f" && rm -f -- "$f" && {
				[ "${arch_f%/}" = "$arch_f" ] && arch_f="~/.ratrun/old/$arch_f"
				[ -n "${verbose-"${default_verbose}"}" ] && printf "${verb_arch_fmt:="$(TEXTDOMAIN=ratrun gettext 'Event "%s" archived in %s\n')"}" "$evname" "$arch_f"
			}
			return
		}
	}
}


parse_config $RERAT_DEFAULT
default_rerat="$rerat"
default_max="$max"
default_archive="$archive"
default_archive_name="$archive_name"
default_verbose="$verbose"


if [ -n "$all_events" ]; then
	set +f
	for f in *; do
		set -f
		do_file
	done
else
	set +f
	grep -sl '^rerat[[:space:]]*:' -- * | {
		set -f
		while read -r f <&3; do
			do_file
		done 3<&0
	}
fi


rm -fd "$PWD" 2>/dev/null  # just in case we moved them all :)
:
