/* SPDX-License-Identifier: MIT */


#include "../main.hpp"
#include "../parse.hpp"
#include "../zfs.hpp"

#include <algorithm>
#include <wchar.h>


#define TZPFMS_BACKEND_MAX_LEN 16


enum class key_loadedness : char {
	none     = -1,
	unloaded = 0,
	loaded   = 1,
};

/// zfs(8) uses struct zprop_get_cbdata_t, which is powerful, but inscrutable; we have a fixed format, which makes this easier
struct output_line {
	static const char * key_available_display[2];
	static const char * coherent_display[2];
	static uint8_t key_available_display_width[2];

	static void init_display();


	char name[ZFS_MAX_DATASET_NAME_LEN + 1];
	char backend[TZPFMS_BACKEND_MAX_LEN + 1];
	bool key_available : 1;
	bool coherent : 1;

	bool included(bool print_nontzpfms, const char * backend_restrixion, key_loadedness key_loadedness_restrixion) const {
		return (print_nontzpfms || !this->coherent || this->backend[0] != '\0') && (!backend_restrixion || !strcmp(backend_restrixion, this->backend)) &&
		       (key_loadedness_restrixion == key_loadedness::none || key_loadedness_restrixion == static_cast<key_loadedness>(this->key_available));
	}

	const char * backend_display() const { return (this->backend[0] != '\0') ? this->backend : "-"; }
};

// for KEYSTATUS column
const char * output_line::key_available_display[2]{gettext_noop("unavailable"), gettext_noop("available")};
// for COHERENT column
const char * output_line::coherent_display[2]{gettext_noop("no"), gettext_noop("yes")};
uint8_t output_line::key_available_display_width[2];


// infallible strings only
static int strwidth(const char * str) {
	int ret{};
	wchar_t c;
	mbstate_t state{};
	auto len = strlen(str);
	for(;;)
		switch(auto rd = mbrtowc(&c, str, len, &state)) {
			case 0:          // NUL
			case(size_t)-1:  // EILSEQ
			case(size_t)-2:  // ENODATA
				return ret;
			default:
				ret += wcwidth(c) ?: 0;
				str += rd;
				len -= rd;
				break;
		}
}

void output_line::init_display() {
	key_available_display[false]       = gettext(key_available_display[false]);
	key_available_display[true]        = gettext(key_available_display[true]);
	key_available_display_width[false] = strwidth(key_available_display[false]);
	key_available_display_width[true]  = strwidth(key_available_display[true]);

	coherent_display[false] = gettext(coherent_display[false]);
	coherent_display[true]  = gettext(coherent_display[true]);
}


int main(int argc, char ** argv) {
	bool human                      = true;
	bool print_nontzpfms            = false;
	size_t maxdepth                 = MAXDEPTH_UNSET;
	const char * backend_restrixion = nullptr;
	auto key_loadedness_restrixion  = key_loadedness::none;
	return do_bare_main(
	    argc, argv, "Hrd:ab:ul", gettext_noop("[-H] [-r|-d max] [-a|-b back-end] [-u|-l]"), gettext_noop("[filesystem|volume]…"),
	    [&](auto arg) {
		    switch(arg) {
			    case 'H':
				    human = false;
				    break;
			    case 'r':
				    maxdepth = SIZE_MAX;
				    break;
			    case 'd':
				    if(!parse_uint(optarg, maxdepth))
					    return fprintf(stderr, "-d %s: %s\n", optarg, strerror(errno)), __LINE__;
				    break;
			    case 'a':
				    print_nontzpfms = true;
				    break;
			    case 'b':
				    backend_restrixion = optarg;
				    break;
			    case 'u':
				    key_loadedness_restrixion = key_loadedness::unloaded;
				    break;
			    case 'l':
				    key_loadedness_restrixion = key_loadedness::loaded;
				    break;
		    }
		    return 0;
	    },
	    [&](auto libz) {
		    output_line * lines{};
		    size_t lines_len{};
		    quickscope_wrapper lines_deleter{[&] { free(lines); }};


		    TRY_MAIN(for_all_datasets(libz, argv + optind, maxdepth, [&](auto dataset) {
			    boolean_t dataset_is_root;
			    TRY("get encryption root", zfs_crypto_get_encryption_root(dataset, &dataset_is_root, nullptr));
			    if(!dataset_is_root)
				    return 0;

			    char *backend{}, *handle{};
			    TRY_MAIN(lookup_userprop(dataset, PROPNAME_BACKEND, backend));
			    TRY_MAIN(lookup_userprop(dataset, PROPNAME_KEY, handle));

			    ++lines_len;
			    lines = TRY_PTR("allocate line buffer", reinterpret_cast<output_line *>(realloc(lines, sizeof(output_line) * lines_len)));

			    auto & cur_line = lines[lines_len - 1];
			    strncpy(cur_line.name, zfs_get_name(dataset), ZFS_MAX_DATASET_NAME_LEN);
			    strncpy(cur_line.backend, (backend && strlen(backend) <= TZPFMS_BACKEND_MAX_LEN) ? backend : "\0", TZPFMS_BACKEND_MAX_LEN);
			    // Tristate available/unavailable/none, but it's gonna be either available or unavailable on envryption roots, so
			    cur_line.key_available = zfs_prop_get_int(dataset, ZFS_PROP_KEYSTATUS) == ZFS_KEYSTATUS_AVAILABLE;
			    cur_line.coherent      = !!backend == !!handle;

			    return 0;
		    }));

		    output_line::init_display();

		    size_t max_name_len          = 0;
		    size_t max_backend_len       = 0;
		    size_t max_key_available_len = 0;
		    auto separator               = "\t";
		    if(human) {
			    max_name_len          = strwidth(gettext("NAME"));
			    max_backend_len       = strwidth(gettext("BACK-END"));
			    max_key_available_len = strwidth(gettext("KEYSTATUS"));
			    separator             = "  ";

			    for(auto cur = lines; cur != lines + lines_len; ++cur)
				    if(cur->included(print_nontzpfms, backend_restrixion, key_loadedness_restrixion)) {
					    max_name_len          = std::max(max_name_len, strlen(cur->name));
					    max_backend_len       = std::max(max_backend_len, strlen(cur->backend_display()));
					    max_key_available_len = std::max(max_key_available_len, static_cast<size_t>(output_line::key_available_display_width[cur->key_available]));
				    }
		    }

		    auto println = [&](auto name, auto backend, auto key_available, auto coherent) {
			    printf("%-*s%s%-*s%s%-*s%s%s\n",                                           //
			           static_cast<int>(max_name_len), name, separator,                    //
			           static_cast<int>(max_backend_len), backend, separator,              //
			           static_cast<int>(max_key_available_len), key_available, separator,  //
			           coherent);
		    };
		    if(human)
			    println(gettext("NAME"), gettext("BACK-END"), gettext("KEYSTATUS"), gettext("COHERENT"));
		    for(auto cur = lines; cur != lines + lines_len; ++cur)
			    if(cur->included(print_nontzpfms, backend_restrixion, key_loadedness_restrixion))
				    println(cur->name, cur->backend_display(), output_line::key_available_display[cur->key_available], output_line::coherent_display[cur->coherent]);

		    return 0;
	    });
}
