/* SPDX-License-Identifier: MIT */


#pragma once


#include "common.hpp"
#include <libintl.h>
#include <libzfs.h>
#include <locale.h>
#include <stdlib.h>
#include <type_traits>
#include <unistd.h>
#define gettext_noop(s) s


#define TRY_PTR(what, ...) TRY_GENERIC(what, !, , errno, __LINE__, strerror, __VA_ARGS__)
#define TRY_MAIN(...)                 \
	do {                                \
		if(auto _try_ret = (__VA_ARGS__)) \
			return _try_ret;                \
	} while(0)


template <class G, class M, class V = int (*)()>
static int do_bare_main(
    int argc, char ** argv, const char * getoptions, const char * usage, const char * dataset_usage, G && getoptfn, M && main,
    V && validate = []() { return 0; }) {
	setlocale(LC_ALL, "");
	bindtextdomain("tzpfms", "/usr/share/locale");
	// bindtextdomain("tzpfms", "out/locale");
	textdomain("tzpfms");

	const auto libz = TRY_PTR("initialise libzfs", libzfs_init());
	libzfs_print_on_error(libz, B_TRUE);

	auto gopts = reinterpret_cast<char *>(alloca(strlen(getoptions) + 2 + 1));
	gopts[0] = 'h', gopts[1] = 'V';
	strcpy(gopts + 2, getoptions);
	for(int opt; (opt = getopt(argc, argv, gopts)) != -1;)
		switch(opt) {
			case '?':
			case 'h':
				fprintf(opt == 'h' ? stdout : stderr, gettext("usage: %s [-hV] %s%s%s\n"), argv[0], *usage ? gettext(usage) : "", *usage ? " " : "",
				        gettext(dataset_usage));
				return opt == 'h' ? 0 : __LINE__;
			case 'V':
				puts("tzpfms version " TZPFMS_VERSION);
				return 0;
			default:
				if constexpr(std::is_same_v<std::invoke_result_t<G, decltype(opt)>, void>)
					getoptfn(opt);
				else if constexpr(std::is_arithmetic_v<std::invoke_result_t<G, decltype(opt)>>)
					TRY_MAIN(getoptfn(opt));
				else {
					if(auto err = getoptfn(opt))
						return fprintf(stderr, gettext("usage: %s [-hV] %s%s%s\n"), argv[0], *usage ? gettext(usage) : "", *usage ? " " : "", gettext(dataset_usage)), err;
				}
		}

	if(auto err = validate())
		return fprintf(stderr, gettext("usage: %s [-hV] %s%s%s\n"), argv[0], *usage ? gettext(usage) : "", *usage ? " " : "", gettext(dataset_usage)), err;
	return main(libz);
}

template <class G, class M, class V = int (*)()>
static int do_main(int argc, char ** argv, const char * getoptions, const char * usage, G && getoptfn, M && main, V && validate = []() { return 0; }) {
	return do_bare_main(
	    // as-in argument in a usage string
	    argc, argv, getoptions, usage, gettext_noop("dataset"), getoptfn,
	    [&](auto libz) {
		    if(!*(argv + optind))
			    return fprintf(stderr,
			                   gettext("No dataset to act on?\n"
			                           "usage: %s [-hV] %s%sdataset\n"),
			                   argv[0], usage, strlen(usage) ? " " : ""),
			           __LINE__;
		    if(*(argv + optind + 1))
			    return fprintf(stderr, gettext("usage: %s [-hV] %s%sdataset\n"), argv[0], usage, strlen(usage) ? " " : ""), __LINE__;
		    auto dataset = TRY_PTR(nullptr, zfs_open(libz, argv[optind], ZFS_TYPE_FILESYSTEM | ZFS_TYPE_VOLUME));
		    quickscope_wrapper dataset_deleter{[&] { zfs_close(dataset); }};

		    {
			    char encryption_root[MAXNAMELEN];
			    boolean_t dataset_is_root;
			    TRY("get encryption root", zfs_crypto_get_encryption_root(dataset, &dataset_is_root, encryption_root));

			    if(!dataset_is_root && !strlen(encryption_root))
				    return fprintf(stderr, gettext("Dataset %s not encrypted?\n"), zfs_get_name(dataset)), __LINE__;
			    else if(!dataset_is_root) {
				    fprintf(stderr, gettext("Using dataset %s's encryption root %s instead.\n"), zfs_get_name(dataset), encryption_root);
				    zfs_close(dataset);
				    dataset = TRY_PTR(nullptr, zfs_open(libz, encryption_root, ZFS_TYPE_FILESYSTEM | ZFS_TYPE_VOLUME));
			    }
		    }


		    return main(dataset);
	    },
	    validate);
}
